'use strict';

const { EnumerationResult } = require('affinity:common')
const { HandleObject } = require('./handleobject.js')
const {
    CubicBezier,
    CurveCornerData,
    CurveCornerType,
    CurveApi,
    CurveBuilderApi,
    CurveNode,
    CurveNodeStyle,
    CurveNodeType,
    Point,
    PolyCurveApi,
    PolygonApi,
    Rectangle,
    RectangleInt,
    Size,
    SizeInt,
    SplineApi,
    SplineProfile,
    Transform,
    TransformData,
    Vector,
    XY
} = require('affinity:geometry');
const { Collection, SpanCollection } = require('./collection.js');

// sorting function for numbers
function numberComp(a, b) {
    return a - b;
}

// returns true IFF the ranges intersect
function rangesIntersect(a1, a2, b1, b2) {
    let a = [a1, a2].sort(numberComp);
    let b = [b1, b2].sort(numberComp);
    return !(a[1] <= b[0] || b[1] <= a[0]);
}

// returns the intersection of two ranges, or nothing if the ranges don't intersect
function intersectRanges(a1, a2, b1, b2) {
	let a = [a1, a2].sort(numberComp)
    let b = [b1, b2].sort(numberComp);
			
    if (a[1] < b[0] || b[1] < a[0]) {
    	return;
    }
    
    let res =
    [
   		Math.max(a[0], b[0]),
    	Math.min(a[1], b[1])
   	];
    return res;
}

// returns true IFF the rectangles overlap
function rectsIntersect(rc1, rc2) {
    let r =
        rangesIntersect(rc1.x, rc1.x + rc1.width, rc2.x, rc2.x + rc2.width) &&
        rangesIntersect(rc1.y, rc1.y + rc1.height, rc2.y, rc2.y + rc2.height);
 return r;
}

// returns the intersection of two rectangles, or nothing if the rectangles don't intersect
function intersectRects(rc1, rc2) {
	let a = intersectRanges(rc1.x, rc1.x + rc1.width, rc2.x, rc2.x + rc2.width);
	if (a) {
		let b = intersectRanges(rc1.y, rc1.y + rc1.height, rc2.y, rc2.y + rc2.height);
		if (b) {
			let res = new Rectangle(
				a[0],
				b[0],
				a[1] - a[0],
				b[1] - b[0],
			);
			return res;
		}
	}
}

function unionRanges(a1, a2, b1, b2) {
	let res = 
	[
		Math.min(a1, b1),
		Math.max(a2, b2)
	];
	return res;
}

function unionRects(rc1, rc2) {
	const xs = unionRanges(rc1.x, rc1.x + rc1.width, rc2.x, rc2.x + rc2.width);
	const ys = unionRanges(rc1.y, rc1.y + rc1.height, rc2.y, rc2.y + rc2.height);
	return new Rectangle(xs[0], ys[0], xs[1] - xs[0], ys[1] - ys[0]);
}

// returns true IFF a value is within a range
function valueInRange(r1, r2, value) {
    let range = [r1, r2].sort(numberComp);
    return value >= range[0] && value <= range[1];
}

// returns true if a points is within a rectangle
function pointInRect(rc, pt) {
    return valueInRange(rc.x, rc.x + rc.width, pt.x) && valueInRange(rc.y, rc.y + rc.height, pt.y);
}


class CurveBuilder extends HandleObject {
	constructor(handle) {
		if (handle === undefined)
			super(CurveBuilderApi.create());
		else
			super(handle);
	}

	static create() {
		return new CurveBuilder(CurveBuilderApi.create());
	}

	get [Symbol.toStringTag]() {
		return 'CurveBuilder';
	}

	clone() {
		return new CurveBuilder(CurveBuilderApi.clone(this.handle));
	}

	begin() {
		CurveBuilderApi.begin(this.handle, ...arguments);
		return this;
	}

	lineTo() {
		CurveBuilderApi.lineTo(this.handle, ...arguments);
		return this;
	}

	lineRel() {
		CurveBuilderApi.lineRel(this.handle, ...arguments);
		return this;
	}

	close() {
		CurveBuilderApi.close(this.handle);
		return this;
	}

	createCurve() {
		const curveHandle = CurveBuilderApi.createCurve(this.handle);
		return new Curve(curveHandle);
	}

	transform(xf) {
		CurveBuilderApi.transform(this.handle, xf);
		return this;
	}

	translate() {
		return this.transform(Transform.createTranslate(...arguments));
	}

	rotate() {
		return this.transform(Transform.createRotate(...arguments));
	}

	scale(x, y) {
		if (y === undefined) {
			y = x;
		}
		return this.transform(Transform.createScale(x, y));
	}
    
    versineTo() {
        CurveBuilderApi.versineTo(this.handle, ...arguments);
        return this;
    }
    
    versineRel() {
        CurveBuilderApi.versineRel(this.handle, ...arguments);
        return this;
    }
    
    addArc(startAngle, endAngle, radius, reverse) {
        CurveBuilderApi.addArc(this.handle, startAngle, endAngle, radius, reverse);
        return this;
    }
    
    addBezier() {
        CurveBuilderApi.addBezier(this.handle, ...arguments);
        return this;
    }
    
    addEllipse(startAngle, endAngle, majorRadius, minorRadius, rotation, reverse) {
        CurveBuilderApi.addEllipse(this.handle, startAngle, endAngle, majorRadius, minorRadius, rotation, reverse);
        return this;
    }
    
    setCornerAt(pointID, cornerData, force) {
        CurveBuilderApi.setCornerAt(this.handle, pointID, cornerData, force);
        return this;
    }
}


class Curve extends HandleObject {
    
    get [Symbol.toStringTag]() {
        return 'Curve';
    }
    
    constructor(handle) {
        super(handle);
    }
    
    static create() {
        return new Curve(CurveApi.create());
    }
    
    static createLine(p1, p2) {
        return new Curve(CurveApi.createLine(p1, p2));
    }
    
    static createRect(rc) {
        return new Curve(CurveApi.createRect(rc));
    }
    
    static createEllipse(rc) {
        return new Curve(CurveApi.createEllipse(rc));
    }
    
    static createDiamond(rc) {
        return new Curve(CurveApi.createEllipse(rc));
    }
    
    static createLozenge(p1, p2, radius) {
        return new Curve(CurveApi.createLozenge(p1, p2, radius));
    }
    
    static createUnitCircle() {
        return new Curve(CurveApi.createUnitCircle());
    }
    
    clone() {
        return new Curve(CurveApi.clone(this.handle));
    }
    
    transform(xf) {
        CurveApi.transform(this.handle, xf);
        return this;
    }
    
    translate() {
        return this.transform(Transform.createTranslate(...arguments));
    }
    
    get pointCount() {
        return CurveApi.getPointCount(this.handle);
    }
    
    get length() {
        return CurveApi.getLength(this.handle);
    }
    
    get isClockwise() {
        return CurveApi.isClockwise(this.handle);
    }
    
    get isEmpty() {
        return CurveApi.isEmpty(this.handle);
    }
    
    get isClosed() {
        return CurveApi.isClosed(this.handle);
    }
    
    get isRectangle() {
        return CurveApi.isRectangle(this.handle);
    }
    
    get isEllipse() {
        return CurveApi.isEllipse(this.handle);
    }
    
    get isStraightLine() {
        return CurveApi.isStraightLine(this.handle);
    }
    
    get isPolyline() {
        return CurveApi.isPolyline(this.handle);
    }
    
    get approxBoundingBox() {
        const rc = CurveApi.getApproxBoundingBox(this.handle);
        return new Rectangle(rc.x, rc.y, rc.width, rc.height);
    }
    
    get exactBoundingBox() {
        const rc = CurveApi.getExactBoundingBox(this.handle);
        return new Rectangle(rc.x, rc.y, rc.width, rc.height);
    }
    
    get path() {
        return CurveApi.getPath(this.handle);
    }
    
    get points() {
        const handle = this.handle
        const span = {
            get length() {
                return CurveApi.getPointCount(handle);
            },
            at(index) {
                return CurveApi.getPointAt(handle, index);
            }
        };
        return new SpanCollection(span);
    }
    
    getPointAt(index) {
        const pt = CurveApi.getPointAt(this.handle, index);
        return new Point(pt.x, pt.y);
    }
    
    // index, pt
    setPointAt() {
        CurveApi.setPointAt(this.handle, ...arguments);
    }
    
    getCornerAt(pointID) {
        return CurveApi.getCornerAt(this.handle, pointID);
    }
    
    setCornerAt(pointID, cornerData, force) {
        CurveApi.setCornerAt(this.handle, pointID, cornerData, force);
        return this;
    }
    
    cut(isParametric, distance) {
        let curves = CurveApi.cut(this.handle, isParametric, distance);
        return { "curveA": new Curve(curves.curveA), "curveB": new Curve(curves.curveB) };
    }
    
    getCubicBezier(pointIndex) {
        return CurveApi.getCubicBezier(this.handle, pointIndex);
    }
    
    get firstOnCurvePointIndex() {
        return CurveApi.getFirstOnCurvePointIndex(this.handle);
    }
    
    getNextOnCurvePointIndex(pointIndex, allowWrapIfClosed) {
        return CurveApi.getNextOnCurvePointIndex(this.handle, pointIndex, allowWrapIfClosed);
    }
    
    getPreviousOnCurvePointIndex(pointIndex, allowWrapIfClosed) {
        return CurveApi.getPreviousOnCurvePointIndex(this.handle, pointIndex, allowWrapIfClosed);
    }
    
    get lastOnCurvePointIndex() {
        return CurveApi.getLastOnCurvePointIndex(this.handle);
    }
    
    get beziers() {
        const me = this;
        const gen = function*() {
            for (let i = me.firstOnCurvePointIndex; i < me.lastOnCurvePointIndex; i = me.getNextOnCurvePointIndex(i)) {
                yield me.getCubicBezier(i);
            }
        }
        return new Collection(gen);
    }

    get nodeCount() {
        return CurveApi.getNodeCount(this.handle);
    }

    getNode(index) {
        return CurveApi.getNode(this.handle, index);
    }

    appendNode(node) {
        CurveApi.appendNode(this.handle, node);
    }

    enumerateNodes(callback) {
        CurveApi.enumerateNodes(this.handle, callback);
    }

    get nodes() {
        let res = [];
        this.enumerateNodes(node => {
            res.push(node);
            return EnumerationResult.Continue;
        });
        return res;
    }

    makeClosed() {
        CurveApi.makeClosed(this.handle);
    }

    generatePolygon(tolerance) {
        return CurveApi.generatePolygon(this.handle, tolerance);
    }
}


class PolyCurve extends HandleObject {
    constructor(handle) {
        if (handle === undefined)
            super(PolyCurveApi.create())
        else
            super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'PolyCurve';
    }

    static create() {
        return new PolyCurve(PolyCurveApi.create());
    }

    clone() {
        return new PolyCurve(PolyCurveApi.clone(this.handle));
    }

    clear() {
        PolyCurveApi.clear(this.handle);
    }

    transform(xf) {
        PolyCurveApi.transform(this.handle, xf);
        return this;
    }

    get curveCount() {
        return PolyCurveApi.getCurveCount(this.handle);
    }

    get approxBoundingBox() {
        return PolyCurveApi.getApproxBoundingBox(this.handle);
    }

    get exactBoundingBox() {
        return PolyCurveApi.getExactBoundingBox(this.handle);
    }

    get controlBoundingBox() {
        return PolyCurveApi.getControlBoundingBox(this.handle);
    }

    at(index) {
        return new Curve(PolyCurveApi.getCurve(this.handle, index));
    }

    *[Symbol.iterator]() {
        for (let i = 0; i < this.curveCount; ++i) {
            yield this.at(i);
        }
    }

    get curves() {
        return new SpanCollection(new PolyCurveCurves(this.handle));
    }
    
    addCurve(curve) {
        return PolyCurveApi.addCurve(this.handle, curve.handle);
    }
}


// Collection for the curves in a PolyCurve
class PolyCurveCurves extends HandleObject {

    constructor(polyCurveHandle) {
        super(polyCurveHandle);
    }

    at(index) {
        return new Curve(PolyCurveApi.getCurve(this.handle, index));
    }

    get length() {
        return PolyCurveApi.getCurveCount(this.handle);
    }
}


class TransformBuilder {
	#transform;
	
	constructor() {
		this.#transform = Transform.createIdentity();
	}
	
	translate(xOrXy, yOrUndefined) {
		this.#transform = Transform.createTranslate(xOrXy, yOrUndefined).multiply(this.#transform);
		return this;
	}
	
	scale(xOrXy, yOrUndefined) {
		this.#transform = Transform.createScale(xOrXy, yOrUndefined).multiply(this.#transform);
		return this;
	}
	
	rotate(rads) {
		this.#transform = Transform.createRotate(rads).multiply(this.#transform);
		return this;
	}
	
	shear(xOrXy, yOrUndefined) {
		this.#transform = Transform.createShear(xOrXy, yOrUndefined).multiply(this.#transform);
		return this;
	}
	
	get transform() {
		return this.#transform;
	}
}

class Spline extends HandleObject {
    
    get [Symbol.toStringTag]() {
        return 'Spline';
    }
    
    constructor(handle) {
        super(handle);
    }
    
    static create() {
        return new Spline(SplineApi.create());
    }
    
    static createFromProfile(profile) {
        return new Spline(SplineApi.createFromProfile(profile));
    }
    
    static createFromPoints(points) {
        return new Spline(SplineApi.createFromPoints(points));
    }
    
    clone() {
        return new Spline(SplineApi.clone(this.handle));
    }
    
    get pointCount() {
        return SplineApi.getPointCount(this.handle);
    }
    
    get linear() {
        return SplineApi.isLinear(this.handle);
    }
    
    set linear(value) {
        return SplineApi.setIsLinear(this.handle, value);
    }
    
    getPoint(index) {
        return SplineApi.getPoint(this.handle, index);
    }
    
    insertPoint(point) {
        return SplineApi.insertPoint(this.handle, point);
    }
    
    replaceOrInsertPoint(point) {
        return SplineApi.replaceOrInsertPoint(this.handle, point);
    }
    
    findPoint(point) {
        return SplineApi.findPoint(this.handle, point);
    }
    
    removePoint(index) {
        return SplineApi.removePoint(this.handle, index);
    }
    
    clear() {
        return SplineApi.clear(this.handle);
    }
    
    *points() {
        const num = this.pointCount;
        for (let i = 0; i < num; i++) {
            yield this.getPoint(i);
        }
    }
}

class Polygon extends HandleObject {
    
    get [Symbol.toStringTag]() {
        return 'Polygon';
    }
    
    constructor(handle) {
        super(handle);
    }
    
    static create() {
        return new Polygon(PolygonApi.create());
    }
    
    static createLine(p1, p2) {
        return new Polygon(PolygonApi.createLine(p1, p2));
    }
    
    static createTriangle(p1, p2, p3) {
        return new Polygon(PolygonApi.createTriangle(p1, p2, p3));
    }
    
    static createRectangle(rect) {
        return new Polygon(PolygonApi.createRectangle(rect));
    }
    
    get isEmpty() {
        return PolygonApi.isEmpty(this.handle);
    }
    
    get isClosed() {
        return PolygonApi.isClosed(this.handle);
    }
    
    get isRectangle() {
        return PolygonApi.isRectangle(this.handle);
    }
    
    get isClockwise() {
        return PolygonApi.isClockwise(this.handle);
    }
    
    enumeratePoints(callback) {
        return PolygonApi.enumeratePoints(this.handle, callback);
    }
    
    clear() {
        return PolygonApi.clear(this.handle);
    }
    
    addPoint(point) {
        return PolygonApi.addPoint(this.handle, point);
    }
    
    insertPoint(point, index) {
        return PolygonApi.insertPoint(this.handle, point, index);
    }
    
    addRectangle(rect) {
        return PolygonApi.addRectangle(this.handle, rect);
    }
    
    addTriangle(p1, p2, p3) {
        return PolygonApi.addTriangle(this.handle, p1, p2, p3);
    }
    
    addLine(p1, p2) {
        return PolygonApi.addLine(this.handle, p1, p2);
    }
    
    close() {
        return PolygonApi.close(this.handle);
    }
    
    reverse() {
        return PolygonApi.reverse(this.handle);
    }
}

module.exports.intersectRanges = intersectRanges;
module.exports.intersectRects = intersectRects;
module.exports.pointInRect = pointInRect;
module.exports.rangesIntersect = rangesIntersect;
module.exports.rectsIntersect = rectsIntersect;
module.exports.unionRanges = unionRanges;
module.exports.unionRects = unionRects;
module.exports.valueInRange = valueInRange;

module.exports.CubicBezier = CubicBezier;
module.exports.Curve = Curve;
module.exports.CurveBuilder = CurveBuilder;
module.exports.CurveCornerData = CurveCornerData;
module.exports.CurveCornerType = CurveCornerType;
module.exports.CurveNode = CurveNode;
module.exports.CurveNodeStyle = CurveNodeStyle;
module.exports.CurveNodeType = CurveNodeType;
module.exports.Point = Point;
module.exports.PolyCurve = PolyCurve;
module.exports.Polygon = Polygon;
module.exports.Rectangle = Rectangle;
module.exports.RectangleInt = RectangleInt;
module.exports.Size = Size;
module.exports.SizeInt = SizeInt;
module.exports.Spline = Spline;
module.exports.SplineProfile = SplineProfile;
module.exports.Transform = Transform;
module.exports.TransformData = TransformData;
module.exports.TransformBuilder = TransformBuilder;
module.exports.Vector = Vector;
