'use strict';

const { DocumentCommandApi } = require('affinity:commands');
const { EnumerationResult } = require('affinity:common')
const { 
    ContentType,
    DocumentApi,
    DocumentExportRecordApi,
    DocumentExportRecordsApi,
    DocumentHistoryApi,
    DocumentHistoryItemApi,
    DocumentPresetApi,
    DocumentSnapshotApi,
    FileExportAreaApi,
    FileExportOptionsApi,
    NewDocumentOptionsApi,
} = require('affinity:dom');
const { Collection } = require('./collection.js');
const { Colour, ColourProfile } = require('./colours.js');
const { DrawingScale } = require('./drawingscale.js');
const { ErrorCode } = require('affinity:common');
const { FillDescriptor, SolidFill } = require('./fills.js');
const { HandleObject } = require('./handleobject.js');
const { LineStyle, LineStyleMask } = require('./linestyle.js');
const { PixelBuffer } = require('./rasterobject.js');
const { Selection, TextSelection } = require('./selections.js');
const { RasterSelection } = require('./rasterselection.js');
const { UnitValueConverter } = require('./units.js');
const Nodes = require('./nodes.js');
const Commands = require('./commands.js');

class DocumentSnapshot extends HandleObject {
    constructor(handle) {
        super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'DocumentSnapshot';
    }

    get description() {
        return DocumentSnapshotApi.getDescription(this.handle);
    }

    get format() {
        return DocumentSnapshotApi.getFormat(this.handle);
    }

    createDocument() {
        return new Document(DocumentApi.createFromSnapshot(this.handle));
    }

    createDocumentAsync(headless, callback) {
        function wrappedCallback(errorCode, documentHandle) {
            let document = documentHandle ? new Document(documentHandle) : null;
            return callback(errorCode, document);
        }
        return DocumentApi.createFromSnapshotAsync(this.handle, false, wrappedCallback);
    }
}


class DocumentPreset extends HandleObject {
    constructor(handle) {
        super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'DocumentPreset';
    }

    static enumerateAll(callback) {
        function wrapped(presetHandle) {
            return callback(new DocumentPreset(presetHandle));
        }
        return DocumentPresetApi.enumerateAll(wrapped);
    }

    static get all() {
        let res = [];
        DocumentPreset.enumerateAll(preset => {
            res.push(preset);
            return EnumerationResult.Continue;
        });
        return res;
    }

    get name() {
        return DocumentPresetApi.getName(this.handle);
    }

    get units() {
        return DocumentPresetApi.getUnits(this.handle);
    }

    get imagePlacement() {
        return DocumentPresetApi.getImagePlacement(this.handle);
    }

    get isTransparentBackground() {
        return DocumentPresetApi.getIsTransparentBackground(this.handle);
    }

    get createArtboard() {
        return DocumentPresetApi.getCreateArtboard(this.handle);
    }

    get isFavourite() {
        return DocumentPresetApi.getIsFavourite(this.handle);
    }

    get facing() {
        return DocumentPresetApi.getFacing(this.handle);
    }

    get verticalStack() {
        return DocumentPresetApi.getVerticalStack(this.handle);
    }

    get doublePageStart() {
        return DocumentPresetApi.getDoublePageStart(this.handle);
    }

    get rasterFormat() {
        return DocumentPresetApi.getRasterFormat(this.handle);
    }

    get width() {
        return DocumentPresetApi.getWidth(this.handle);
    }

    get height() {
        return DocumentPresetApi.getHeight(this.handle);
    }

    get dpi() {
        return DocumentPresetApi.getDpi(this.handle);
    }

    get viewDpi() {
        return DocumentPresetApi.getViewDpi(this.handle);
    }

    get margins() {
        return DocumentPresetApi.getMargins(this.handle);
    }

    get marginsEnabled() {
        return DocumentPresetApi.getMarginsEnabled(this.handle);
    }

    get bleed() {
        return DocumentPresetApi.getBleed(this.handle);
    }

    get useDrawingScale() {
        return DocumentPresetApi.getUseDrawingScale(this.handle);
    }

    get drawingScale() {
        return new DrawingScale(DocumentPresetApi.getDrawingScale(this.handle));
    }

    get colourProfileName() {
        return DocumentPresetApi.getColourProfileName(this.handle);
    }
}

class Layers extends Collection {    
    constructor(documentNodeHandle, reverse, all) {
        const genFunc = function*() {
            for (const spread of Nodes.getNodeChildren(documentNodeHandle, Nodes.NodeChildType.Main, reverse)) {
                const f = all ? Nodes.getNodeChildrenRecursive : Nodes.getNodeChildren;
                for (const node of f(spread.handle, Nodes.NodeChildType.Main, reverse)) {
                    yield node;
                }
            }
        };
        super(genFunc);

        this.reverse = function() {
            return new Layers(documentNodeHandle, !reverse, all);
        }

        Object.defineProperty(this, "all", {
            get: function() {
                return new Layers(documentNodeHandle, reverse, true);
            }
        });
    }
}


class Document extends HandleObject {
    constructor(handle) {
        super(handle)
    }

    get [Symbol.toStringTag]() {
        return 'Document';
    }

    get promises() {
        return new DocumentPromises(this.handle);
    }

    isSameObject(otherDocument) {
        return DocumentApi.isSameObject(this.handle, otherDocument.handle);
    }

    get isOpen() {
        return DocumentApi.isOpen(this.handle);
    }

    get isHeadless() {
        return DocumentApi.isHeadless(this.handle);
    }

    get path() {
        return DocumentApi.getPath(this.handle);
    }

    #rootNode;
    get rootNode() {
        if (!this.#rootNode)
            this.#rootNode = new Nodes.DocumentNode(DocumentApi.getRootNode(this.handle));
        return this.#rootNode;
    }

    get selection() {
        return new Selection(DocumentApi.getCurrentSelection(this.handle));
    }

    set selection(items) {
        if (items) {
            if (items.isSelection) {
                const cmd = DocumentCommandApi.createSetSelectionCommand(items.handle);
                DocumentApi.executeCommand(this.handle, cmd);
            }
            else {
                this.selection = Selection.create(this, items);
            }
        }
        else {
            const cmd = DocumentCommandApi.createSetSelectionCommand(Selection.createEmpty(this).handle);
            DocumentApi.executeCommand(this.handle, cmd);
        }
    }

    get dpi() {
        return DocumentApi.getDpi(this.handle);
    }

    get viewdpi() {
        return DocumentApi.getViewDpi(this.handle);
    }

    executeCommand(documentCommand, preview) {
        return DocumentApi.executeCommand(this.handle, documentCommand.handle, preview);
    }

    #history
    get history() {
        if (!this.#history)
            this.#history = DocumentHistory.create(this);
        return this.#history;
    }

    static get current() {
        let res = DocumentApi.getCurrent();
        if (res && DocumentApi.isOpen(res))
            res = new Document(res);
        else
            res = null;
        return res;
    }

    static enumerateAll(callback) {
        function wrapped(documentHandle) {
            return callback(new Document(documentHandle));
        }
        return DocumentApi.enumerateOpen(wrapped);
    }

    static get all() {
        const res = [];
        function callback(document) {
            res.push(document);
            return EnumerationResult.Continue;
        }
        Document.enumerateAll(callback);
        return res;
    }

    static load(path) {
        return new Document(DocumentApi.load(path));
    }

    get isEmbedded() {
        return DocumentApi.isEmbedded(this.handle);
    }

    get isReadOnly() {
        return DocumentApi.isReadOnly(this.handle);
    }

    get mustSaveAs() {
        return DocumentApi.mustSaveAs(this.handle);
    }

    get needsSaving() {
        return DocumentApi.needsSaving(this.handle);
    }

    get isDirty() {
        return DocumentApi.isDirty(this.handle);
    }

    get title() {
        return DocumentApi.getTitle(this.handle);
    }

    save() {
        return DocumentApi.save(this.handle);
    }

    saveAs(path) {
        return DocumentApi.saveAs(this.handle, path);
    }

    saveAsPackage(path, includeFonts, includeImages) {
        return DocumentApi.saveAsPackage(this.handle, path, includeFonts, includeImages);
    }

    get colourProfile() {
        return new ColourProfile(DocumentApi.getColourProfile(this.handle));
    }

    get currentSpread() {
        return new Nodes.SpreadNode(DocumentApi.getCurrentSpread(this.handle));
    }

    close() {
        return DocumentApi.close(this.handle);
    }

    static createFromPreset(preset, landscape) {
        if (landscape == null)
            landscape = preset.width > preset.height;
        return new Document(DocumentApi.createFromPreset(preset.handle, landscape));
    }

    static createFromOptions(options) {
        return new Document(DocumentApi.createFromOptions(options.handle));
    }

    static create(options) {
        return new Document(DocumentApi.createFromOptions(options.handle));
    }

    export(path, exportOptions, exportArea, size) {
        return new DocumentExportRecords(DocumentApi.export(this.handle, path, exportOptions.handle, exportArea?.handle, size));
    }

    get format() {
        return DocumentApi.getFormat(this.handle);
    }

    get maskFormat() {
        return DocumentApi.getMaskFormat(this.handle);
    }

    get units() {
        return DocumentApi.getUnits(this.handle);
    }

    get unitValueConverter() {
        return new UnitValueConverter(DocumentApi.getUnitValueConverter(this.handle));
    }

    enumerateSnapshots(callback) {
        function wrapped(handle) {
            return callback(new DocumentSnapshot(handle));
        }
        return DocumentApi.enumerateSnapshots(this.handle, wrapped);
    }

    get snapshotCount() {
        return DocumentApi.getSnapshotCount(this.handle);
    }

    get snapshots() {
        const res = [];
        function callback(snapshot) {
            res.push(snapshot);
            return EnumerationResult.Continue;
        }
        this.enumerateSnapshots(callback);
        return res;
    }
    
    get currentSnapshotIndex() {
        return DocumentApi.getCurrentSnapshotIndex(this.handle);
    }
    
    get currentSnapshotHistoryIndex() {
        return DocumentApi.getCurrentSnapshotHistoryIndex(this.handle);
    }
    
    get currentSnapshot() {
        return new DocumentSnapshot(DocumentApi.getCurrentSnapshot(this.handle));
    }

    get rasterSelection() {
        return new RasterSelection(DocumentApi.getRasterSelection(this.handle));
    }

    enumerateFontNames(callback) {
        return DocumentApi.enumerateFontNames(this.handle, callback);
    }
    
    getFontNames() {
        let names = [];
        this.enumerateFontNames((name, isInstalled) => { 
            names.push(name); 
            return EnumerationResult.Continue;
        });
        return names;
    }
    
    // async

    #checkOptionalCallback(callback) {
        if (callback == null || callback == undefined)
            return null;
        if (callback instanceof Function)
            return callback;
        throw "INVALID_ARGS";
    }

    static getCurrentAsync(callback) {
        function wrapped(errorCode, documentHandle) {
            const document = documentHandle ? new Document(documentHandle) : null;
            callback(errorCode, document);
        }
        return DocumentApi.getCurrentAsync(wrapped);
    }
    
    static loadAsync(path, options, headless, callback) {
        function wrapped(errorCode, documentHandle) {
            const document = documentHandle ? new Document(documentHandle) : null;
            callback(errorCode, document);
        }
        return DocumentApi.loadAsync(path, options, headless, wrapped);
    }

    static createFromPresetAsync(preset, callback, headless, landscape) {
        function wrapped(errorCode, documentHandle) {
            const document = documentHandle ? new Document(documentHandle) : null;
            callback(errorCode, document);
        }
        if (landscape == null) {
            landscape = preset.width > preset.height;
        }
        return DocumentApi.createFromPresetAsync(this.handle, landscape, headless, wrapped);
    }

    static createAsync(options, callback) {
        function wrapped(errorCode, documentHandle) {
            const document = documentHandle ? new Document(documentHandle) : null;
            callback(errorCode, document);
        }
        return DocumentApi.createFromOptionsAsync(this.handle, options.handle, wrapped);
    }

    executeCommandAsync(command, callback, preview) {
        return DocumentApi.executeCommandAsync(this.handle, command.handle, callback, preview);
    }

    saveAsync(callback) {
        return DocumentApi.saveAsync(this.handle, callback);
    }

    saveAsAsync(path, callback) {
        return DocumentApi.saveAsAsync(this.handle, path, callback);
    }

    saveAsPackageAsync(path, includeFonts, includeImages, callback) {
        return DocumentApi.saveAsPackageAsync(this.handle, path, includeFonts, includeImages, callback);
    }

    closeAsync(callback) {
        return DocumentApi.closeAsync(this.handle, callback);
    }

    exportAsync(path, exportOptions, exportArea, size, callback) {
        callback = this.#checkOptionalCallback(callback);
        if (!callback)
            return DocumentApi.exportAsync(this.handle, path, exportOptions.handle, exportArea?.handle, size, null);
        function wrapped(errorCode, documentExportRecordsHandle) {
            const documentExportRecords = documentExportRecordsHandle ? new DocumentExportRecords(documentExportRecordsHandle) : null;
            callback(errorCode, documentExportRecords);
        }
        return DocumentApi.exportAsync(this.handle, path, exportOptions.handle, exportArea?.handle, size, wrapped);
    }

    // helpers
    get sizePixels() {
        const bbox = this.rootNode.baseBoxInterface.baseBox;
        return {
            width: bbox.width,
            height: bbox.height,
        };
    }

    get widthPixels() {
        const bbox = this.rootNode.baseBoxInterface.baseBox;
        return bbox.width;
    }

    get heightPixels() {
        const bbox = this.rootNode.baseBoxInterface.baseBox;
        return bbox.height;
    }

    get layers() {
        return new Layers(this.rootNode.handle, false, false);
    }

    get spreads() {
        return new Nodes.NodeChildrenOnly(this.rootNode.handle, Nodes.NodeChildType.Main, false);
    }
    
    get artboards() {
        return this.spreads.first.artboards;
    }

    get canUndo() {
        return this.history.canUndo;
    }

    get canRedo() {
        return this.history.canRedo;
    }

    get undoDescription() {
        return this.history.undoDescription;
    }

    get redoDescription() {
        return this.history.redoDescription;
    }

    // synchronous command helpers
    #ensureSelection(selection) {
        if (selection == null || selection.isSelection)
            return selection;
        else
            return Selection.create(this, selection);
    }

    #makeFillDescriptor(fillDescriptor) {
        if (fillDescriptor == null) {
            return FillDescriptor.createNone();
        }
        if (fillDescriptor instanceof FillDescriptor) {
            return fillDescriptor;
        }
        if (fillDescriptor instanceof SolidFill) {
            return FillDescriptor.createSolid(fillDescriptor);
        }
        if (fillDescriptor instanceof Colour) {
            return FillDescriptor.createSolid(fillDescriptor);
        }
        return fillDescriptor;
    }
    undo() {
        this.history.undo();
    }

    redo() {
        this.history.redo();
    }

    selectAll(selectOnCurrentLayerOnly, preview) {
        const cmd = Commands.DocumentCommand.createSelectAll(selectOnCurrentLayerOnly);
        return this.executeCommand(cmd, preview);
    }

    deleteSelection(selection, preview) {
        const cmd = Commands.DocumentCommand.createDeleteSelection(this.#ensureSelection(selection));
        return this.executeCommand(cmd, preview);
    }

    setEditable(editable, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetEditable(this.#ensureSelection(selection), editable);
        return this.executeCommand(cmd, preview);
    }

    lockSelection(selection, preview) {
        return this.setEditable(false, selection, preview);
    }
    
    unlockSelection(selection, preview) {
        return this.setEditable(true, selection, preview);
    }

    setVisible(visible, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetVisibility(this.#ensureSelection(selection), visible);
        return this.executeCommand(cmd, preview);
    }

    showSelection(selection, preview) {
        return this.setVisible(true, selection, preview);
    }

    hideSelection(selection, preview) {
        return this.setVisible(false, selection, preview);
    }

    showAll(preview) {
        const cmd = Commands.DocumentCommand.createShowAll();
        return this.executeCommand(cmd, preview);
    }

    setBrushFillDescriptor(fillDescriptorOrColour, selection, options = null, preview = false) {
        const cmd = Commands.DocumentCommand.createSetBrushFill(this.#ensureSelection(selection), this.#makeFillDescriptor(fillDescriptorOrColour), options);
        return this.executeCommand(cmd, preview);
    }

    setPenFillDescriptor(fillDescriptorOrColour, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetPenFill(this.#ensureSelection(selection), this.#makeFillDescriptor(fillDescriptorOrColour));
        return this.executeCommand(cmd, preview);
    }

    setTransparencyFillDescriptor(fillDescriptorOrColour, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetTransparencyFill(this.#ensureSelection(selection), this.#makeFillDescriptor(fillDescriptorOrColour));
        return this.executeCommand(cmd, preview);
    }

    setLineStyleDescriptor(lineStyleDescriptor, selection, options = null, preview = false) {
        const cmd = Commands.DocumentCommand.createSetLineStyleDescriptor(this.#ensureSelection(selection), lineStyleDescriptor, options);
        return this.executeCommand(cmd, preview);
    }

    setLineStyle(lineStyle, selection, options = null, preview = false) {
        const cmd = Commands.DocumentCommand.createSetLineStyle(this.#ensureSelection(selection), lineStyle, options);
        return this.executeCommand(cmd, preview);
    }

    setLineCap(cap, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        lineStyle.cap = cap;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.Cap}, preview);
    }

    setLineJoin(join, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        lineStyle.join = join;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.Join}, preview);
    }

    setLineType(type, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        lineStyle.type = type;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.Type}, preview);
    }

    setLineWeight(weight, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        lineStyle.weight = weight;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.Weight}, preview);
    }

    setLineWeightPts(weightInPts, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        const pointsToPixels = this.dpi / 72;
        lineStyle.weight = weightInPts * pointsToPixels;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.Weight}, preview);
    }

    setDashPattern(dashPattern, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        lineStyle.dashPattern = dashPattern;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.DashPattern}, preview);
    }

    setDashPhase(dashPhase, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        lineStyle.dashPhase = dashPhase;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.DashPhase}, preview);
    }

    setBalancedDashes(balanced, selection, preview) {
        const lineStyle = LineStyle.createDefault();
        lineStyle.hasBalancedDashes = balanced;
        return this.setLineStyle(lineStyle, this.#ensureSelection(selection), {lineStyleMask: LineStyleMask.BalancedDashes}, preview);
    }

    setStrokeAlignment(strokeAlignment, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetStrokeAlignment(this.#ensureSelection(selection), strokeAlignment);
        return this.executeCommand(cmd, preview);
    }

    importMacro(path) {
        const cmd = Commands.DocumentCommand.createImportMacro(path);
        return this.executeCommand(cmd);
    }

    exportMacro(path) {
        const cmd = Commands.DocumentCommand.createExportMacro(path);
        return this.executeCommand(cmd);
    }

    clearMacro() {
        const cmd = Commands.DocumentCommand.createClearMacro();
        return this.executeCommand(cmd);
    }

    startRecordingMacro() {
        const cmd = Commands.DocumentCommand.createStartRecordingMacro();
        return this.executeCommand(cmd);
    }

    stopRecordingMacro() {
        const cmd = Commands.DocumentCommand.createStopRecordingMacro();
        return this.executeCommand(cmd);
    }

    addGuide(horizontal, pixels96, preview) {
        const cmd = Commands.DocumentCommand.createAddGuide(horizontal, pixels96);
        return this.executeCommand(cmd, preview);
    }

    moveGuide(horizontal, index, newPixels96, preview) {
        const cmd = Commands.DocumentCommand.createMoveGuide(horizontal, index, newPixels96);
        return this.executeCommand(cmd, preview);
    }

    removeGuide(horizontal, index, preview) {
        const cmd = Commands.DocumentCommand.createRemoveGuide(horizontal, index);
        return this.executeCommand(cmd, preview);
    }

    setGuidesColour(colour, preview) {
        const cmd = Commands.DocumentCommand.createSetGuidesColour(colour);
        return this.executeCommand(cmd, preview);
    }

    setLayerDescription(description, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetDescription(this.#ensureSelection(selection), description);
        return this.executeCommand(cmd, preview);
    }

    setTagColour(colour, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetTagColour(this.#ensureSelection(selection), colour);
        return this.executeCommand(cmd, preview);
    }

    setShape(shape, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetShape(this.#ensureSelection(selection), shape, options);
        return this.executeCommand(cmd, preview);
    }

    setShapeFloatParam(key, value, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetShapeFloatParam(this.#ensureSelection(selection), key, value, options);
        return this.executeCommand(cmd, preview);
    }

    setShapeIntParam(key, value, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetShapeIntParam(this.#ensureSelection(selection), key, value);
        return this.executeCommand(cmd, preview);
    }

    setShapeBoolParam(key, value, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetShapeBoolParam(this.#ensureSelection(selection), key, value);
        return this.executeCommand(cmd, preview);
    }

    setShapeEnumParam(key, value, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetShapeEnumParam(this.#ensureSelection(selection), key, value);
        return this.executeCommand(cmd, preview);
    }

    setBrushFillIsAnchoredToSpread(anchoredToSpread, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetBrushFillIsAnchoredToSpread(this.#ensureSelection(selection), anchoredToSpread, options);
        return this.executeCommand(cmd, preview);
    }

    setPenFillIsAnchoredToSpread(anchoredToSpread, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetPenFillIsAnchoredToSpread(this.#ensureSelection(selection), anchoredToSpread, options);
        return this.executeCommand(cmd, preview);
    }

    setTransparencyFillIsAnchoredToSpread(anchoredToSpread, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetTransparencyFillIsAnchoredToSpread(this.#ensureSelection(selection), anchoredToSpread, options);
        return this.executeCommand(cmd, preview);
    }
    
    rasteriseObjects(selection, rasteriseContentsOnly, clipToSpread, preview) {
        const cmd = Commands.DocumentCommand.createRasteriseObjects(this.#ensureSelection(selection), rasteriseContentsOnly, clipToSpread);
        return this.executeCommand(cmd, preview);
    }
    
    convertToCurves(selection, preview) {
        const cmd = Commands.DocumentCommand.createConvertToCurves(this.#ensureSelection(selection));
        return this.executeCommand(cmd, preview);
    }

    setBrushHatchFillAttributes(attr, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetBrushHatchFillAttributes(this.#ensureSelection(selection), attr, options);
        return this.executeCommand(cmd, preview);
    }

    setPenHatchFillAttributes(attr, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetPenHatchFillAttributes(this.#ensureSelection(selection), attr, options);
        return this.executeCommand(cmd, preview);
    }

    setTransparencyHatchFillAttributes(attr, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createSetTransparencyHatchFillAttributes(this.#ensureSelection(selection), attr, options);
        return this.executeCommand(cmd, preview);
    }

    clearPreviews() {
        const cmd = Commands.DocumentCommand.createClearPreviews();
        return this.executeCommand(cmd);
    }

    applyTransform(transform, selection, options, preview) {
        const cmd = Commands.DocumentCommand.createTransform(this.#ensureSelection(selection), transform, options);
        return this.executeCommand(cmd, preview);
    }

    setText(text, selection, preview) {
        const cmd = Commands.DocumentCommand.createSetText(this.#ensureSelection(selection), text);
        return this.executeCommand(cmd, preview);
    }

    insertGlyph(glyph, selection, preview) {
        const cmd = Commands.DocumentCommand.createInsertGlyph(this.#ensureSelection(selection), glyph);
        return this.executeCommand(cmd, preview);
    }

    insertGlyphAt(glyph, textNode, position, preview) {
        const selection = Selection.create(this, textNode);
        const textSelection = TextSelection.create([{begin:position, end:position}]);
        selection.addSubSelection(textNode, textSelection);
        const cmd = Commands.DocumentCommand.createInsertGlyph(selection, glyph);
        return this.executeCommand(cmd, preview);
    }
    
    rasterSelectAll(preview){
        const cmd = Commands.DocumentCommand.createRasterSelectAll();
        return this.executeCommand(cmd, preview);
    }
    
    rasterDeselect(preview){
        const cmd = Commands.DocumentCommand.createRasterDeselect();
        return this.executeCommand(cmd, preview);
    }
    
    rasterInvertSelection(preview){
        const cmd = Commands.DocumentCommand.createRasterInvertSelection();
        return this.executeCommand(cmd, preview);
    }
    
    rasterReselect(preview){
        const cmd = Commands.DocumentCommand.createRasterReselect();
        return this.executeCommand(cmd, preview);
    }

    rasterSelectPolygon(polygon, operation, isAntialias, featherRadius, preview) {
        const cmd = Commands.DocumentCommand.createRasterSelectPolygon(polygon, operation, isAntialias, featherRadius);
        return this.executeCommand(cmd, preview);
    }

    // TODO: asynchronous command helpers
    setLayerDescriptionAsync(description, selection, callback, preview) {
        const cmd = Commands.DocumentCommand.createSetDescription(this.#ensureSelection(selection), description);
        return this.executeCommandAsync(cmd, callback, preview);
    }

    setTagColourAsync(colour, selection, callback, preview) {
        const cmd = Commands.DocumentCommand.createSetTagColour(this.#ensureSelection(selection), colour);
        return this.executeCommandAsync(cmd, callback, preview);
    }

    setShapeAsync(shape, selection, callback, preview) {
        const cmd = Commands.DocumentCommand.createSetShape(this.#ensureSelection(selection), shape);
        return this.executeCommandAsync(cmd, callback, preview);
    }

    clearPreviewsAsync(callback) {
        const cmd = Commands.DocumentCommand.createClearPreviews();
        return this.executeCommandAsync(cmd, callback);
    }

    get pageCount() {
        return this.rootNode.pageCount;
    }

    get spreadCount() {
        return this.rootNode.spreadCount;
    }
}


class NewDocumentOptions extends HandleObject {
    constructor(handle) {
        super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'NewDocumentOptions';
    }

    static createDefault() {
        return new NewDocumentOptions(NewDocumentOptionsApi.createDefault());
    }
    
    static createFromPreset(documentPreset) {
        return new NewDocumentOptions(NewDocumentOptionsApi.createFromPreset(documentPreset.handle));
    }
    
    static getMaxDpi(unit, dimension) {
        return NewDocumentOptionsApi.getMaxDpi(unit, dimension);
    }
    
    static getMaxSize(unit, dpi) {
        return NewDocumentOptionsApi.getMaxSize(unit, dpi);
    }
    
    set units(unit) {
        NewDocumentOptionsApi.setUnits(this.handle, unit);
    }
    
    set imagePlacement(placement) {
        NewDocumentOptionsApi.setImagePlacement(this.handle, placement);
    }
    
    set isTransparentBackground(isTransparent) {
        NewDocumentOptionsApi.setIsTransparentBackground(this.handle, isTransparent);
    }
    
    set createArtboard(createArtboard) {
        NewDocumentOptionsApi.setCreateArtboard(this.handle, createArtboard);
    }
    
    set isFacing(facing) {
        NewDocumentOptionsApi.setIsFacing(this.handle, facing);
    }
    
    set isLandscape(isLandscape) {
        NewDocumentOptionsApi.setIsLandscape(this.handle, isLandscape);
    }
    
    set isVerticalStack(verticalStack) {
        NewDocumentOptionsApi.setIsVerticalStack(this.handle, verticalStack);
    }
    
    set isDoublePageStart(doublePageStart) {
        NewDocumentOptionsApi.setIsDoublePageStart(this.handle, doublePageStart);
    }
    
    set rasterFormat(format) {
        NewDocumentOptionsApi.setRasterFormat(this.handle, format);
    }
    
    set colourProfile(profile) {
        NewDocumentOptionsApi.setColourProfile(this.handle, profile);
    }
    
    set width(w) {
        NewDocumentOptionsApi.setWidth(this.handle, w);
    }
    
    set height(h) {
        NewDocumentOptionsApi.setHeight(this.handle, h);
    }
    
    set dpi(dpi) {
        NewDocumentOptionsApi.setDpi(this.handle, dpi);
    }
    
    set viewDpi(dpi) {
        NewDocumentOptionsApi.setViewDpi(this.handle, dpi);
    }
    
    set margins(margins) {
        NewDocumentOptionsApi.setMargins(this.handle, margins);
    }
    
    set marginsEnabled(enabled) {
        NewDocumentOptionsApi.setMarginsEnabled(this.handle, enabled);
    }
    
    set bleed(bleed) {
        NewDocumentOptionsApi.setBleed(this.handle, bleed);
    }
    
    set pageCount(pCount) {
        NewDocumentOptionsApi.setPageCount(this.handle, pCount);
    }
    
    set createMaster(create) {
        NewDocumentOptionsApi.setCreateMaster(this.handle, create);
    }
    
    set useDrawingScale(use) {
        NewDocumentOptionsApi.setUseDrawingScale(this.handle, use);
    }
    
    set drawingScale(drawingScale) {
        NewDocumentOptionsApi.setDrawingScale(this.handle, drawingScale.handle);
    }
}

class DocumentHistoryItem extends HandleObject {
    constructor(handle) {
        super(handle)
    }

    get [Symbol.toStringTag]() {
        return 'DocumentHistoryItem';
    }

    get hasAlternateFutures() {
        return DocumentHistoryItemApi.hasAlternateFutures(this.handle);
    }

    get command() {
        return new DocumentCommand(DocumentHistoryItemApi.getCommand(this.handle));
    }

    get thumbnail() {
        return new PixelBuffer(DocumentHistoryItemApi.getThumbnail(this.handle));
    }

    get description() {
        return DocumentHistoryItemApi.getDescription(this.handle);
    }

    dispose() {
        DocumentHistoryItemApi.dispose();
    }
}


// note this takes a document handle
class DocumentHistory extends HandleObject {
    #documentHandle;

    constructor(documentHandle) {
        super(DocumentApi.getHistory(documentHandle));
        this.#documentHandle = documentHandle;
    }

    static create(document) {
        const history = new DocumentHistory(document.handle);
        return history;
    }

    get documentHandle() {
        return this.#documentHandle;
    }

    get document() {
        return new Document(this);
    }

    get [Symbol.toStringTag]() {
        return 'DocumentHistory';
    }

    get position() {
        return DocumentHistoryApi.getUndoPosition(this.handle);
    }

    set position(index) {
        const commandHandle = DocumentCommandApi.createSetHistoryIndexCommand(index - 1);
        DocumentApi.executeCommand(this.#documentHandle, commandHandle);
    }

    get size() {
        return DocumentHistoryApi.getSize(this.handle);
    }

    get canUndo() {
        return DocumentHistoryApi.canUndo(this.handle);
    }

    get canRedo() {
        return DocumentHistoryApi.canRedo(this.handle);
    }

    get undoDescription() {
        return DocumentHistoryApi.getUndoDescription(this.handle);
    }

    get redoDescription() {
        return DocumentHistoryApi.getRedoDescription(this.handle);
    }

    getItem(index) {
        let itemHandle = DocumentHistoryApi.getItem(this.handle, index);
        if (itemHandle) {
            return new DocumentHistoryItem(itemHandle);
        }
        else {
            return null;
        }
    }

    enumerateItems(callback) {
        function wrapped(documentHistoryItemHandle) {
            return callback(new DocumentHistoryItem(documentHistoryItemHandle));
        }
        return DocumentHistoryApi.enumerateItems(this.handle, wrapped);
    }

    get items() {
        let items = [];
        this.enumerateItems((item) => {
            items.push(item);
            return false;
        });
        return items;
    }

    undo(preview) {
        const commandHandle = DocumentCommandApi.createUndoCommand();
        DocumentApi.executeCommand(this.documentHandle, commandHandle, preview);
    }

    redo(preview) {
        const commandHandle = DocumentCommandApi.createRedoCommand();
        DocumentApi.executeCommand(this.documentHandle, commandHandle, preview);
    }

    undoAsync(callback, preview) {
        const commandHandle = DocumentCommandApi.createUndoCommand();
        DocumentApi.executeCommandAsync(this.documentHandle, commandHandle, callback, preview);
    }

    redoAsync(callback, preview) {
        const commandHandle = DocumentCommandApi.createRedoCommand();
        DocumentApi.executeCommandAsync(this.documentHandle, commandHandle, callback, preview);
    }
}

class DocumentExportRecord extends HandleObject {
    constructor(handle) {
        super(handle)
    }

    get [Symbol.toStringTag]() {
        return 'DocumentExportRecord';
    }

    get path() {
        return DocumentExportRecordApi.getPath(this.handle);
    }

    get hasWarnings() {
        return DocumentExportRecordApi.hasWarnings(this.handle);
    }

    get warningString() {
        return DocumentExportRecordApi.getWarningString(this.handle);
    }

    get warningMessage() {
        return DocumentExportRecordApi.getWarningMessage(this.handle);
    }

    get errorMessage() {
        return DocumentExportRecordApi.getErrorMessage(this.handle);
    }

    get result() {
        return DocumentExportRecordApi.getResult(this.handle);
    }

    get isSuccess() {
        const result_value = this.result.value;
        return result_value == ErrorCode.OK || result_value == ErrorCode.WARNINGS_ONLY;
    }
}

class DocumentExportRecords extends HandleObject {
    constructor(handle) {
        super(handle)
    }

    get [Symbol.toStringTag]() {
        return 'DocumentExportRecords';
    }

    get count() {
        return DocumentExportRecordsApi.getCount(this.handle);
    }

    enumerate(callback) {
        return DocumentExportRecordsApi.enumerate(this.handle,
            (documentExportRecordHandle) => { return callback(new DocumentExportRecord(documentExportRecordHandle)); });
    }

    get all() {
        let records = [];
        this.enumerate(record => {
            records.push(record);
            return false;
        });
        return records;
    }
}

class FileExportOptions extends HandleObject {
    constructor(handle) {
        super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'FileExportOptions';
    }

    static enumeratePresetNames(callback) {
        return FileExportOptionsApi.enumeratePresetNames(callback);
    }

    static createWithPresetName(presetName) {
        return new FileExportOptions(FileExportOptionsApi.createWithPresetName(presetName));
    }

    static createForCanvaExport(dpi) {
        return new FileExportOptions(FileExportOptionsApi.createForCanvaExport(dpi));
    }

    static get allPresetNames() {
        let presetNames = [];
        FileExportOptions.enumeratePresetNames(presetName => {
            presetNames.push(presetName);
            return EnumerationResult.Continue;
        });
        return presetNames;
    }
}

class FileExportArea extends HandleObject {
    constructor(handle) {
        super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'FileExportArea';
    }

    clone() {
        return new FileExportArea(FileExportAreaApi.clone(this.handle));
    }

    static createWithWholeDocument() {
        return new FileExportArea(FileExportAreaApi.createWithWholeDocument());
    }

    static createWithCurrentSpread() {
        return new FileExportArea(FileExportAreaApi.createWithCurrentSpread());
    }

    static createWithCurrentPage() {
        return new FileExportArea(FileExportAreaApi.createWithCurrentPage());
    }

    static createWithArtboard(artboardInterface) {
        return new FileExportArea(FileExportAreaApi.createWithArtboard(artboardInterface.handle));
    }

    static createWithSpreads(strPages) {
        return new FileExportArea(FileExportAreaApi.createWithSpreads(strPages));
    }

    static createWithPages(strPages) {
        return new FileExportArea(FileExportAreaApi.createWithPages(strPages));
    }

    static createWithSelection(selection) {
        return new FileExportArea(FileExportAreaApi.createWithSelection(selection?.handle));
    }

    static createWithSelectionArea(selection) {
        return new FileExportArea(FileExportAreaApi.createWithSelectionArea(selection?.handle));
    }
}

class DocumentPromises extends HandleObject {
    constructor(handle) {
        super(handle)
    }

    get document() {
        return new Document(this.handle);
    }

    // TODO: implement
}


module.exports.Document = Document;
module.exports.DocumentExportRecord = DocumentExportRecord;
module.exports.DocumentExportRecords = DocumentExportRecords;
module.exports.DocumentPreset = DocumentPreset;
module.exports.DocumentSnapshot = DocumentSnapshot;
module.exports.FileExportArea = FileExportArea;
module.exports.FileExportOptions = FileExportOptions;
module.exports.NewDocumentOptions = NewDocumentOptions;
