'use strict';

const { EnumerationResult } = require('affinity:common')
const { SelectionApi, SelectionItemApi, SubSelectionApi, SubSelectionType, TextSelectionApi } = require('affinity:dom');
const { Collection, SpanCollection } = require("./collection.js");
const { HandleObject } = require("./handleobject.js");
const Nodes = require("./nodes.js");

function createTypedSubSelection(subSelectionHandle) {
	switch (SubSelectionApi.getSubSelectionType(subSelectionHandle).value) {
		case SubSelectionType.Text.value:
			return new TextSelection(TextSelectionApi.fromSubSelection(subSelectionHandle));
		default:
			return new SubSelection(subSelectionHandle);
	}
}

class SelectionItem extends HandleObject {
	
	constructor(handle) {
		super(handle);
	}

	get [Symbol.toStringTag]() {
		return 'SelectionItem';
	}
	
	static get isSelectionItem() {
		return true;
	}

	get node() {
		const nodeHandle = SelectionItemApi.getNode(this.handle);
		if (!nodeHandle)
			return null;
		return Nodes.createTypedNode(nodeHandle);
	}

	getSubSelection(index) {
		return createTypedSubSelection(SelectionItemApi.getSubSelection(this.handle, index));
	}

	getSubSelectionOfType(subSelectionType) {
		const handle = SelectionItemApi.getSubSelectionOfType(this.handle, subSelectionType);
		return handle ? createTypedSubSelection(handle) : handle;
	}

	get subSelectionCount() {
		return SelectionItemApi.getSubSelectionCount(this.handle, );
	}

	enumerateSubSelections(callback) {
		function wrapped(subSelectionHandle) {
            return callback(createTypedSubSelection(subSelectionHandle));
        }
        return SelectionItemApi.enumerateSubSelections(this.handle, wrapped);
	}

	get subSelections() {
		const res = [];
        this.enumerateSubSelections(subSelection => {
            res.push(subSelection);
            return EnumerationResult.Continue;
        });
        return res;
	}
}

class Selection extends HandleObject {
    constructor(handle) {
        if (!handle)
            super(SelectionApi.createEmpty())
        else
            super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'Selection';
    }

    get length() {
        return SelectionApi.getCount(this.handle);
    }

    at(index) {
        return new SelectionItem(SelectionApi.getItem(this.handle, index));
    }

    get items() {
        return new SpanCollection(this);
    }

	get nodes() {
		return this.items.map(item => item.node).filter(node => node);
	}

	get firstNode() {
		return this.nodes.first;
	}
    
    add(nodeOrItem) {
        if (nodeOrItem.isNode) {
            this.addNode(nodeOrItem);
        } else if (nodeOrItem.isSelectionItem) {
            this.addItem(nodeOrItem);
        } else {
            throw new TypeError("expected Node or SelectionItem");
        }
    }

	addNode(node) {
		return SelectionApi.addNode(this.handle, node.handle);
	}
    
    addItem(item) {
        return SelectionApi.addItem(this.handle, item.handle);
    }
	
	addSubSelectionForNode(node, subSelection) {
		return SelectionApi.addSubSelectionForNode(this.handle, node.handle, subSelection.handle);
	}
	
	get isSelection() {
		return true;
	}

	getFirstSubSelectionOfType(subSelectionType) {
		const handle = SelectionApi.getFirstSubSelectionOfType(this.handle, subSelectionType);
		return handle ? createTypedSubSelection(handle) : null;
	}

	removeNested() {
		return SelectionApi.removeNested(this.handle);
	}

	containsItem(item) {
		return SelectionApi.containsItem(this.handle, item.handle);
	}
	
	static create(document, items, removeNested) {
		let sel = Selection.createEmpty(document);
		if (items != null) {
			if (items.isNode || items.isSelectionItem) {
				sel.add(items);
			}
			else if (items[Symbol.iterator]) {
				for (const item of items) {
					sel.add(item);
				}
			}
			if (removeNested)
				sel.removeNested();
		}
		return sel;
	}

	static createEmpty(document) {
		return new Selection(SelectionApi.createEmpty(document.handle));
	}
}

class SubSelection extends HandleObject {
    constructor(handle) {
        super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'SubSelection';
    }

    static get isSubSelection() {
        return true;
    }

    isSame(other) {
        return SubSelectionApi.isSame(this.handle, other.handle);
    }

	get subSelectionType() {
		return SubSelectionApi.getSubSelectionType(this.handle);
	}
}


class TextSelection extends SubSelection {
	constructor(handle) {
		super(handle);
	}

	get [Symbol.toStringTag]() {
		return 'TextSelection';
	}

	static create(rangesOrNull) {
		if (rangesOrNull != null) {
			if (rangesOrNull instanceof Collection)
				rangesOrNull = rangesOrNull.toArray();
			else if (!(rangesOrNull instanceof Array))
				rangesOrNull = [rangesOrNull];
		}
		return new TextSelection(TextSelectionApi.create(rangesOrNull));
	}
	
	get isEmpty() {
		return TextSelectionApi.isEmpty(this.handle);
	}

	get hasMarkedText() {
		return TextSelectionApi.hasMarkedText(this.handle);
	}

	get caret() {
		return TextSelectionApi.getCaret(this.handle);
	}

	get anchor() {
		return TextSelectionApi.getAnchor(this.handle);
	}

	get markedTextBegin() {
		return TextSelectionApi.getMarkedTextBegin(this.handle);
	}

	get markedTextEnd() {
		return TextSelectionApi.getMarkedTextEnd(this.handle);
	}

	get rangeCount() {
		return TextSelectionApi.getRangeCount(this.handle);
	}

	getRange(index) {
		return TextSelectionApi.getRange(this.handle, index);
	}

	enumerateRanges(callback) {
		return TextSelectionApi.enumerateRanges(this.handle, callback);
	}

	get ranges() {
		let res = [];
        function callback(range) {
            res.push(range);
            return EnumerationResult.Continue;
        }
        this.enumerateRanges(callback);
        return res;
	}
};


module.exports.Selection = Selection;
module.exports.SelectionItem = SelectionItem;
module.exports.SubSelection = SubSelection;
module.exports.SubSelectionType = SubSelectionType;
module.exports.TextSelection = TextSelection;
