'use strict';

const { CommandApi, DocumentCommandApi, CompoundCommandBuilderApi, AddChildNodesCommandBuilderApi, InsertionMode, SetHatchFillAttributesCommandBuilderApi } = require('affinity:commands');
const { ContentType, NodeMoveType, NodeChildType, VisibilityMode, RasterSelectionLogicalOperation } = require('affinity:dom');
const { Colour } = require('./colours.js');
const { FillMask, FillDescriptor, SolidFill } = require('./fills.js');
const { Transform } = require('./geometry.js');
const { HandleObject } = require('./handleobject.js');
const Nodes = require("./nodes.js");

class Command extends HandleObject {
    constructor(handle) {
        super(handle);
    }

    get [Symbol.toStringTag]() {
        return 'Command';
    }

    get description() {
        return CommandApi.getDescription(this.handle);
    }
}

class DocumentCommand extends HandleObject {
    constructor(handle) {
        super(handle);
    }
    
    get [Symbol.toStringTag]() {
        return 'DocumentCommand';
    }
    
    get result() {
        return DocumentCommandApi.getResult(this.handle);
    }

    get description() {
        return CommandApi.getDescription(this.handle);
    }

    static createSetSelection(selection) {
        return new DocumentCommand(DocumentCommandApi.createSetSelectionCommand(selection?.handle));
    }
    
    static createDeleteSelection(selection, ignoreRasterSelection) {
        return new DocumentCommand(DocumentCommandApi.createDeleteNodesCommand(selection?.handle, ignoreRasterSelection));
    }
    
    static createHideSelection(selection) {
        return new DocumentCommand(DocumentCommandApi.createSetVisibilityCommand(selection?.handle, VisibilityMode.Main, false));
    }
    
    static createTransform(selection, xf, options) {
        if (!xf)
            xf = new Transform();
        return new DocumentCommand(DocumentCommandApi.createTransformCommand(
            selection?.handle,
            xf,
            options?.mergeable,
            options?.cloneRaster,
            options?.correctChildren,
            options?.duplicateNodes)
        );
    }

    static createSetBrushFill(selection, fillDescriptorOrColour, options) {
        if (fillDescriptorOrColour instanceof Colour || fillDescriptorOrColour instanceof SolidFill)
            fillDescriptorOrColour = FillDescriptor.createSolid(fillDescriptorOrColour);
        return new DocumentCommand(DocumentCommandApi.createSetBrushFillCommand(
            selection?.handle,
            fillDescriptorOrColour?.handle,
            options?.contentType ?? ContentType.Main,
            options?.fillMask ?? FillMask.All,
            options?.copyTransform ?? true,
            options?.useTextSelection,
            options?.maintainBlendMode,
            options?.restoreAspectRatio)
        );
    }

    static createSetPenFill(selection, fillDescriptorOrColour, options) {
        if (fillDescriptorOrColour instanceof Colour || fillDescriptorOrColour instanceof SolidFill)
            fillDescriptorOrColour = FillDescriptor.createSolid(fillDescriptorOrColour);
        return new DocumentCommand(DocumentCommandApi.createSetPenFillCommand(
            selection?.handle,
            fillDescriptorOrColour?.handle,
            options?.contentType ?? ContentType.Main,
            options?.fillMask ?? FillMask.All,
            options?.copyTransform ?? true,
            options?.useTextSelection,
            options?.maintainBlendMode,
            options?.restoreAspectRatio)
        );
    }

    static createSetTransparencyFill(selection, fillDescriptorOrColour, options) {
        if (fillDescriptorOrColour instanceof Colour || fillDescriptorOrColour instanceof SolidFill)
            fillDescriptorOrColour = FillDescriptor.createSolid(fillDescriptorOrColour);
        return new DocumentCommand(DocumentCommandApi.createSetTransparencyFillCommand(
            selection?.handle,
            fillDescriptorOrColour?.handle,
            options?.contentType ?? ContentType.Main,
            options?.fillMask ?? FillMask.All,
            options?.copyTransform ?? true,
            options?.useTextSelection,
            options?.maintainBlendMode,
            options?.restoreAspectRatio)
        );
    }

    static createSetVisibility(selection, visible, options) {
        return new DocumentCommand(DocumentCommandApi.createSetVisibilityCommand(
            selection?.handle,
            options?.visibilityMode ?? VisibilityMode.Main,
            visible)
        );
    }

    static createSetLineStyle(selection, lineStyle, options) {
        return new DocumentCommand(DocumentCommandApi.createSetLineStyleCommand(
            selection?.handle,
            lineStyle.handle,
            options?.lineStyleMask,
            options?.contentType,
            options?.useTextSelection,
            options?.defaultsMode)
        );
    }

    static createSetLineStyleDescriptor(selection, lineStyleDescriptor, options) {
        return new DocumentCommand(DocumentCommandApi.createSetLineStyleDescriptorCommand(
            selection?.handle,
            lineStyleDescriptor.handle,
            options?.lineStyleMask,
            options?.contentType,
            options?.useTextSelection,
            options?.defaultsMode)
        );
    }

    static createSetStrokeAlignment(selection, strokeAlignment, options) {
        return new DocumentCommand(DocumentCommandApi.createSetStrokeAlignmentCommand(
            selection?.handle,
            strokeAlignment,
            options?.contentType,
            options?.useTextSelection,
            options?.defaultsMode)
        );
    }
    
    static createMoveNodes(selection, target, moveType, childListType) {
        return new DocumentCommand(DocumentCommandApi.createMoveNodesCommand(
            selection?.handle, 
            target.handle,
            moveType,
            childListType)
        );
    }

    static createUndo() {
        return new DocumentCommand(DocumentCommandApi.createUndoCommand());
    }

    static createRedo() {
        return new DocumentCommand(DocumentCommandApi.createRedoCommand());
    }

    static createSetHistoryIndex(index) {
        return new DocumentCommand(DocumentCommandApi.createSetHistoryIndexCommand(index));
    }

    static createCycleAlternateFutures(index) {
        return new DocumentCommand(DocumentCommandApi.createCycleAlternateFuturesCommand(index));
    }

    static createSetShape(selection, shape, options) {
        return new DocumentCommand(DocumentCommandApi.createSetShapeCommand(selection?.handle, shape.handle, options?.allowReplaceLikeShapes));
    }

    static createSetShapeFloatParam(selection, key, value, options) {
        return new DocumentCommand(DocumentCommandApi.createSetShapeFloatParamCommand(selection?.handle, key, value, options?.constrained, options?.symmetrical));
    }

    static createSetShapeIntParam(selection, key, value) {
        return new DocumentCommand(DocumentCommandApi.createSetShapeIntParamCommand(selection?.handle, key, value));
    }

    static createSetShapeBoolParam(selection, key, value) {
        return new DocumentCommand(DocumentCommandApi.createSetShapeBoolParamCommand(selection?.handle, key, value));
    }

    static createSetShapeEnumParam(selection, key, value) {
        return new DocumentCommand(DocumentCommandApi.createSetShapeEnumParamCommand(selection?.handle, key, value));
    }

    static createShowAll() {
        return new DocumentCommand(DocumentCommandApi.createShowAllCommand());
    }

    static createSelectAll(selectOnCurrentLayerOnly) {
        return new DocumentCommand(DocumentCommandApi.createSelectAllCommand(selectOnCurrentLayerOnly));
    }

    static createSetText(selection, text) {
        return new DocumentCommand(DocumentCommandApi.createSetTextCommand(selection?.handle, text));
    }

    static createInsertGlyph(selection, glyph) {
        return new DocumentCommand(DocumentCommandApi.createInsertGlyphCommand(selection?.handle, glyph.handle));
    }

    static createSetTagColour(selection, colour) {
        return new DocumentCommand(DocumentCommandApi.createSetTagColourCommand(selection?.handle, colour.handle));
    }

    static createAddCurve(curve) {
        return new AddNodeCommand(DocumentCommandApi.createAddCurveCommand(curve.handle));
    }

    static createSetDescription(selection, description) {
        return new DocumentCommand(DocumentCommandApi.createSetDescriptionCommand(selection?.handle, description));
    }
    
    static createAutosave() {
        return new DocumentCommand(DocumentCommandApi.createAutosaveCommand());
    }
    
    static createSetEmbeddedDocumentLayerVisibility(selection, layer, visibility) {
        return new DocumentCommand(DocumentCommandApi.createSetEmbeddedDocumentLayerVisibilityCommand(selection?.handle, layer, visibility));
    }
    
    static createSetEmbeddedDocumentAllLayersVisibility(selection, visibilities) {
        return new DocumentCommand(DocumentCommandApi.createSetEmbeddedDocumentAllLayersVisibilityCommand(selection?.handle, visibilities));
    }
    
    static createSetEmbeddedDocumentSelectedArtboardID(selection, artboardID) {
        return new DocumentCommand(DocumentCommandApi.createSetEmbeddedDocumentSelectedArtboardIDCommand(selection?.handle, artboardID));
    }
    
    static createSetEmbeddedDocumentSelectedSpreadID(selection, spreadID) {
        return new DocumentCommand(DocumentCommandApi.createSetEmbeddedDocumentSelectedArtboardIDCommand(selection?.handle, spreadID));
    }
    
    static createSetEmbeddedDocumentPageBoundingBoxType(selection, pageBoundingBoxType) {
        return new DocumentCommand(DocumentCommandApi.createSetEmbeddedDocumentPageBoundingBoxTypeCommand(selection?.handle, pageBoundingBoxType));
    }
    
    static createSetEmbeddedDocumentPDFPassthrough(selection, isPDFPassthrough) {
        return new DocumentCommand(DocumentCommandApi.createAutosaveCommand(selection?.handle, isPDFPassthrough));
    }

    static createKnifeCut(cutCurve, selection) {
        return new DocumentCommand(DocumentCommandApi.createKnifeCutCommand(cutCurve.handle, selection?.handle));
    }

    static createScissorCut(node, polyCurveIndex, curveIndex, isParametric, distance) {
        return new DocumentCommand(DocumentCommandApi.createScissorCutCommand(node.handle, polyCurveIndex, curveIndex, isParametric, distance));
    }

    static createSplitCurve(selection, parametricDistance) {
        return new DocumentCommand(DocumentCommandApi.createSplitCurveCommand(selection?.handle, parametricDistance));
    }
    
    static createImportMacro(path) {
        return new DocumentCommand(DocumentCommandApi.createImportMacroCommand(path));
    }
    
    static createExportMacro(path) {
        return new DocumentCommand(DocumentCommandApi.createExportMacroCommand(path));
    }
    
    static createReplayMacro() {
        return new DocumentCommand(DocumentCommandApi.createReplayMacroCommand());
    }
    
    static createClearMacro() {
        return new DocumentCommand(DocumentCommandApi.createClearMacroCommand());
    }
    
    static createStartRecordingMacro() {
        return new DocumentCommand(DocumentCommandApi.createStartRecordingMacroCommand());
    }
    
    static createStopRecordingMacro() {
        return new DocumentCommand(DocumentCommandApi.createStopRecordingMacroCommand());
    }

    static createSetEditable(editable, changeSelection) {
        return new DocumentCommand(DocumentCommandApi.createSetEditableCommand(editable, changeSelection));
    }

    static createSetDocumentUnits(units) {
        return new DocumentCommand(DocumentCommandApi.createSetDocumentUnitsCommand(units));
    }

    static createAddGuide(horizontal, pixels96) {
        return new DocumentCommand(DocumentCommandApi.createAddGuideCommand(horizontal, pixels96));
    }

    static createMoveGuide(horizontal, index, newPixels96) {
        return new DocumentCommand(DocumentCommandApi.createAddGuideCommand(horizontal, index, newPixels96));
    }

    static createRemoveGuide(horizontal, index) {
        return new DocumentCommand(DocumentCommandApi.createRemoveGuideCommand(horizontal, index));
    }

    static createSetGuidesColour(colour) {
        return new DocumentCommand(DocumentCommandApi.createSetGuidesColourCommand(colour.handle));
    }

    static createRotateCanvas(clockwise) {
        return new DocumentCommand(DocumentCommandApi.createRotateCanvasCommand(clockwise));
    }
    
    static createRasteriseObjects(selection, rasteriseContentsOnly, clipToSpread) {
        return new DocumentCommand(DocumentCommandApi.createRasteriseObjectsCommand(selection?.handle, rasteriseContentsOnly, clipToSpread));
    }
    
    static createConvertToCurves(selection) {
        return new DocumentCommand(DocumentCommandApi.createConvertToCurvesCommand(selection?.handle));
    }

    static createSetBrushFillIsAnchoredToSpread(selection, anchoredToSpread, options) {
        return new DocumentCommand(DocumentCommandApi.createSetBrushFillIsAnchoredToSpreadCommand(
            selection?.handle,
            anchoredToSpread,
            options?.contentType,
            options?.useTextSelection,
            options?.applyToAllFills
        ));
    }

    static createSetPenFillIsAnchoredToSpread(selection, anchoredToSpread, options) {
        return new DocumentCommand(DocumentCommandApi.createSetPenFillIsAnchoredToSpreadCommand(
            selection?.handle,
            anchoredToSpread,
            options?.contentType,
            options?.useTextSelection,
            options?.applyToAllFills
        ));
    }

    static createSetTransparencyFillIsAnchoredToSpread(selection, anchoredToSpread, options) {
        return new DocumentCommand(DocumentCommandApi.createSetTransparencyFillIsAnchoredToSpreadCommand(
            selection?.handle,
            anchoredToSpread,
            options?.contentType,
            options?.useTextSelection,
            options?.applyToAllFills
        ));
    }

    static #createHatchFillAttributesCommandBuilder(attr) {
        const builder = SetHatchFillAttributesCommandBuilder.create();
        if (attr.pattern !== undefined)
            builder.pattern = attr.pattern;
        if (attr.penColour !== undefined)
            builder.penColour = attr.penColour;
        if (attr.brushColour !== undefined)
            builder.brushColour = attr.brushColour;
        if (attr.scale !== undefined)
            builder.scale = attr.scale;
        if (attr.rotation !== undefined)
            builder.rotation = attr.rotation;
        if (attr.lineWeight !== undefined)
            builder.lineWeight = attr.lineWeight;
        if (attr.fixScale !== undefined)
            builder.fixScale = attr.fixScale;
        if (attr.useRelativeTransform !== undefined)
            builder.useRelativeTransform = attr.useRelativeTransform;
        if (attr.useCurrentItemOriginForRelativeTransform !== undefined)
            builder.useCurrentItemOriginForRelativeTransform = attr.useCurrentItemOriginForRelativeTransform;
        return builder;
    }
    
    static createSetBrushHatchFillAttributes(selection, attr, options) {
        const builder = DocumentCommand.#createHatchFillAttributesCommandBuilder(attr);
        return builder.createBrushFillCommand(selection,
            options?.contentType ?? ContentType.Main,
            options?.useTextSelection,
            options?.applyToAllFills);
    }

    static setPenHatchFillAttributes(attr, selection, options) {
        const builder = DocumentCommand.#createHatchFillAttributesCommandBuilder(attr);
        return builder.createPenFillCommand(selection,
            options?.contentType ?? ContentType.Main,
            options?.useTextSelection,
            options?.applyToAllFills);
    }

    static setTransparencyHatchFillAttributes(attr, selection, options) {
        const builder = DocumentCommand.#createHatchFillAttributesCommandBuilder(attr);
        return builder.createTransparencyFillCommand(selection,
            options?.contentType ?? ContentType.Main,
            options?.useTextSelection,
            options?.applyToAllFills);
    }
    
    static createReplaceBitmap(selection, bitmap) {
        return new DocumentCommand(DocumentCommandApi.createReplaceBitmapCommand(selection?.handle, bitmap.handle));
    }

    static createSetPolyCurveNodeCurves(polyCurveNode, polyCurve) {
        return new DocumentCommand(DocumentCommandApi.createSetPolyCurveNodeCurvesCommand(polyCurveNode.handle, polyCurve.handle));
    }

    static createClearPreviews() {
        return new DocumentCommand(DocumentCommandApi.createClearPreviewsCommand());
    }
    
    static createAddDocumentSnapshot(description) {
        return new DocumentCommand(DocumentCommandApi.createAddDocumentSnapshotCommand(description));
    }

    static createDeleteDocumentSnapshot(snapshot) {
        return new DocumentCommand(DocumentCommandApi.createDeleteDocumentSnapshotCommand(snapshot.handle));
    }

    static createRestoreDocumentSnapshot(snapshot) {
        return new DocumentCommand(DocumentCommandApi.createRestoreDocumentSnapshotCommand(snapshot.handle));
    }

    static createSetCurrentSnapshot(snapshot) {
        return new DocumentCommand(DocumentCommandApi.createSetCurrentSnapshotCommand(snapshot.handle));
    }

    static createSetCurrentSnapshotFromHistoryItem(historyItem) {
        return new DocumentCommand(DocumentCommandApi.createSetCurrentSnapshotFromHistoryItemCommand(historyItem.handle));
    }

    static createSetGaussianBlurFilterParameters(selection, gaussianBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetGaussianBlurFilterParametersCommand(selection?.handle, gaussianBlurParameters));
    }
    
    static createSetBilateralBlurFilterParameters(selection, bilateralBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetBilateralBlurFilterParametersCommand(selection?.handle, bilateralBlurParameters));
    }
    
    static createSetBoxBlurFilterParameters(selection, boxBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetBoxBlurFilterParametersCommand(selection?.handle, boxBlurParameters));
    }
    
    static createSetMedianBlurFilterParameters(selection, medianBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetMedianBlurFilterParametersCommand(selection?.handle, medianBlurParameters));
    }
    
    static createSetDiffuseGlowFilterParameters(selection, diffuseGlowParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetDiffuseGlowFilterParametersCommand(selection?.handle, diffuseGlowParameters));
    }
    
    static createSetFieldBlurFilterParameters(selection, fieldBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetFieldBlurFilterParametersCommand(selection?.handle, fieldBlurParameters.handle));
    }
    
    static createSetLensBlurFilterParameters(selection, lensBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetLensBlurFilterParametersCommand(selection?.handle, lensBlurParameters));
    }
    
    static createSetMaximumBlurFilterParameters(selection, maximumBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetMaximumBlurFilterParametersCommand(selection?.handle, maximumBlurParameters));
    }
    
    static createSetMinimumBlurFilterParameters(selection, minimumBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetMinimumBlurFilterParametersCommand(selection?.handle, minimumBlurParameters));
    }
    
    static createSetMotionBlurFilterParameters(selection, motionBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetMotionBlurFilterParametersCommand(selection?.handle, motionBlurParameters));
    }
    
    static createSetRadialBlurFilterParameters(selection, radialBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetRadialBlurFilterParametersCommand(selection?.handle, radialBlurParameters));
    }
    
    static createSetExposureAdjustmentParameters(selection, exposureAdjustmentParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetExposureAdjustmentParametersCommand(selection?.handle, exposureAdjustmentParameters));
    }
    
    static createSetLevelsAdjustmentParameters(selection, levelsAdjustmentParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetLevelsAdjustmentParametersCommand(selection?.handle, levelsAdjustmentParameters));
    }
    
    static createSetLevelsAdjustmentColourSpace(selection, colourSpace) {
        return new DocumentCommand(DocumentCommandApi.createSetLevelsAdjustmentColourSpaceCommand(selection?.handle, colourSpace));
    }
    
    static createSetBrightnessContrastAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetBrightnessContrastAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetShadowsHighlightsAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetShadowsHighlightsAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetBlackAndWhiteAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetBlackAndWhiteAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetRecolourAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetRecolourAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetPosteriseAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetPosteriseAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetSplitToningAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetSplitToningAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetThresholdAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetThresholdAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetClarityFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetClarityFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetUnsharpMaskFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetUnsharpMaskFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetHighPassFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetHighPassFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetDenoiseFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetDenoiseFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetDiffuseFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetDiffuseFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetDustAndScratchFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetDustAndScratchFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetAddNoiseFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetAddNoiseFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetRippleFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetRippleFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetTwirlFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetTwirlFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetSphericalFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetSphericalFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetPinchPunchFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetPinchPunchFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetWhiteBalanceAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetWhiteBalanceAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetColourBalanceAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetColourBalanceAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetVibranceAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetVibranceAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetNormalsAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetNormalsAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetVignetteFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetVignetteFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetDefringeFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetDefringeFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetVoronoiFilterParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetVoronoiFilterParametersCommand(selection?.handle, params));
    }
    
    static createSetSelectiveColourAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetSelectiveColourAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetHSLShiftAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetHSLShiftAdjustmentParametersCommand(selection?.handle, params.handle));
    }
    
    static createSetCurvesAdjustmentParameters(selection, curvesAdjustmentParameters) {
        return new DocumentCommand(DocumentCommandApi.createSetCurvesAdjustmentParametersCommand(selection?.handle, curvesAdjustmentParameters.handle));
    }
    
    static createSetCurvesAdjustmentColourSpace(selection, colourSpace) {
        return new DocumentCommand(DocumentCommandApi.createSetCurvesAdjustmentColourSpaceCommand(selection?.handle, colourSpace));
    }
    
    static createSetTagValueForKey(selection, key, value) {
        return new DocumentCommand(DocumentCommandApi.createSetTagValueForKeyCommand(selection?.handle, key, value));
    }
    
    static createSetTagValueForPredefinedKey(selection, predefinedKey, value) {
        return new DocumentCommand(DocumentCommandApi.createSetTagValueForPredefinedKeyCommand(selection?.handle, predefinedKey, value));
    }
    
    static createSetToneCompressionAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetToneCompressionAdjustmentParametersCommand(selection?.handle, params));
    }
    
    static createSetToneStretchAdjustmentParameters(selection, params) {
        return new DocumentCommand(DocumentCommandApi.createSetToneStretchAdjustmentParametersCommand(selection?.handle, params));
    }

    static createRasterSelectAll() {
        return new DocumentCommand(DocumentCommandApi.createRasterSelectAllCommand());
    }

    static createRasterDeselect() {
        return new DocumentCommand(DocumentCommandApi.createRasterDeselectCommand());
    }

    static createRasterInvertSelection() {
        return new DocumentCommand(DocumentCommandApi.createRasterInvertSelectionCommand());
    }

    static createRasterReselect() {
        return new DocumentCommand(DocumentCommandApi.createRasterReselectCommand());
    }

    static createRasterSelectPolygon(polygon, operation, isAntialias, featherRadius) {
        return new DocumentCommand(DocumentCommandApi.createRasterSelectPolygonCommand(polygon, operation, isAntialias, featherRadius));
    }
    
    static createGaussianBlurFilter(selection, gaussianBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createGaussianBlurFilterCommand(selection?.handle, gaussianBlurParameters));
    }

    static createBoxBlurFilter(selection, boxBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createBoxBlurFilterCommand(selection?.handle, boxBlurParameters));
    }

    static createBilateralBlurFilter(selection, bilateralBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createBilateralBlurFilterCommand(selection?.handle, bilateralBlurParameters));
    }

    static createMedianBlurFilter(selection, medianBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createMedianBlurFilterCommand(selection?.handle, medianBlurParameters));
    }

    static createDiffuseGlowFilter(selection, diffuseGlowParameters) {
        return new DocumentCommand(DocumentCommandApi.createDiffuseGlowFilterCommand(selection?.handle, diffuseGlowParameters));
    }

    static createFieldBlurFilter(selection, fieldBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createFieldBlurFilterCommand(selection?.handle, fieldBlurParameters));
    }

    static createLensBlurFilter(selection, lensBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createLensBlurFilterCommand(selection?.handle, lensBlurParameters));
    }

    static createMaximumBlurFilter(selection, maximumBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createMaximumBlurFilterCommand(selection?.handle, maximumBlurParameters));
    }

    static createMinimumBlurFilter(selection, minimumBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createMinimumBlurFilterCommand(selection?.handle, minimumBlurParameters));
    }

    static createMotionBlurFilter(selection, motionBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createMotionBlurFilterCommand(selection?.handle, motionBlurParameters));
    }

    static createRadialBlurFilter(selection, radialBlurParameters) {
        return new DocumentCommand(DocumentCommandApi.createRadialBlurFilterCommand(selection?.handle, radialBlurParameters));
    }

    static createClarityFilter(selection, clarityFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createClarityFilterCommand(selection?.handle, clarityFilterParameters));
    }

    static createUnsharpMaskFilter(selection, unsharpMaskFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createUnsharpMaskFilterCommand(selection?.handle, unsharpMaskFilterParameters));
    }

    static createHighPassFilter(selection, highPassFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createHighPassFilterCommand(selection?.handle, highPassFilterParameters));
    }

    static createDenoiseFilter(selection, denoiseFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createDenoiseFilterCommand(selection?.handle, denoiseFilterParameters));
    }

    static createDiffuseFilter(selection, diffuseFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createDiffuseFilterCommand(selection?.handle, diffuseFilterParameters));
    }

    static createDustAndScratchFilter(selection, dustAndScratchFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createDustAndScratchFilterCommand(selection?.handle, dustAndScratchFilterParameters));
    }

    static createAddNoiseFilter(selection, addNoiseFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createAddNoiseFilterCommand(selection?.handle, addNoiseFilterParameters));
    }

    static createRippleFilter(selection, rippleFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createRippleFilterCommand(selection?.handle, rippleFilterParameters));
    }

    static createTwirlFilter(selection, twirlFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createTwirlFilterCommand(selection?.handle, twirlFilterParameters));
    }

    static createSphericalFilter(selection, sphericalFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createSphericalFilterCommand(selection?.handle, sphericalFilterParameters));
    }

    static createPinchPunchFilter(selection, pinchPunchFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createPinchPunchFilterCommand(selection?.handle, pinchPunchFilterParameters));
    }

    static createVignetteFilter(selection, vignetteFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createVignetteFilterCommand(selection?.handle, vignetteFilterParameters));
    }

    static createDefringeFilter(selection, defringeFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createDefringeFilterCommand(selection?.handle, defringeFilterParameters));
    }

    static createVoronoiFilter(selection, voronoiFilterParameters) {
        return new DocumentCommand(DocumentCommandApi.createVoronoiFilterCommand(selection?.handle, voronoiFilterParameters));
    }
}


class AddNodeCommand extends DocumentCommand {
    get result() {
        const nodeHandle = DocumentCommandApi.getResult(this.handle);
        return Nodes.createTypedNode(nodeHandle);
    }
}


class CompoundCommandBuilder extends HandleObject {
    constructor(handle) {
        super(handle);
    }
    
    clear() {
        CompoundCommandBuilderApi.clear(this.handle);
    }
    
    addCommand(command, useAsDescription) {
        CompoundCommandBuilderApi.addCommand(this.handle, command.handle, useAsDescription);
        return this;
    }
    
    createCommand() {
        return new DocumentCommand(CompoundCommandBuilderApi.createCommand(this.handle));
    }
    
    static create() {
        return new CompoundCommandBuilder(CompoundCommandBuilderApi.create());
    }
}

class AddChildNodesCommandBuilder extends HandleObject {
    constructor(handle) {
        super(handle);
    }
    
    clearNodes() {
        AddChildNodesCommandBuilderApi.clearNodes(this.handle);
    }
    
    addNode(nodeDefinition) {
        AddChildNodesCommandBuilderApi.addNode(this.handle, nodeDefinition.handle);
    }
    
    addImageNode(imageNodeDefinition) {
        AddChildNodesCommandBuilderApi.addImageNode(this.handle, imageNodeDefinition.handle);
    }
    
    addShapeNode(shapeNodeDefinition) {
        AddChildNodesCommandBuilderApi.addShapeNode(this.handle, shapeNodeDefinition.handle);
    }
    
    addPolyCurveNode(polyCurveNodeDefinition) {
        AddChildNodesCommandBuilderApi.addPolyCurveNode(this.handle, polyCurveNodeDefinition.handle);
    }
    
    addRasterNode(rasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addRasterNode(this.handle, rasterNodeDefinition.handle);
    }
    
    addExposureAdjustmentRasterNode(exposureAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addExposureAdjustmentRasterNode(this.handle, exposureAdjustmentRasterNodeDefinition.handle);
    }
    
    addGaussianBlurFilterRasterNode(gaussianBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addGaussianBlurFilterRasterNode(this.handle, gaussianBlurFilterRasterNodeDefinition.handle);
    }

    addBoxBlurFilterRasterNode(boxBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addBoxBlurFilterRasterNode(this.handle, boxBlurFilterRasterNodeDefinition.handle);
    }
    
    addBilateralBlurFilterRasterNode(bilateralBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addBilateralBlurFilterRasterNode(this.handle, bilateralBlurFilterRasterNodeDefinition.handle);
    }
    
    addMedianBlurFilterRasterNode(medianBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addMedianBlurFilterRasterNode(this.handle, medianBlurFilterRasterNodeDefinition.handle);
    }
    
    addFieldBlurFilterRasterNode(fieldBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addFieldBlurFilterRasterNode(this.handle, fieldBlurFilterRasterNodeDefinition.handle);
    }
    
    addDiffuseGlowFilterRasterNode(diffuseGlowFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addDiffuseGlowFilterRasterNode(this.handle, diffuseGlowFilterRasterNodeDefinition.handle);
    }
    
    addLensBlurFilterRasterNode(lensBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addLensBlurFilterRasterNode(this.handle, lensBlurFilterRasterNodeDefinition.handle);
    }
    
    addMaximumBlurFilterRasterNode(maximumBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addMaximumBlurFilterRasterNode(this.handle, maximumBlurFilterRasterNodeDefinition.handle);
    }
    
    addMinimumBlurFilterRasterNode(minimumBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addMinimumBlurFilterRasterNode(this.handle, minimumBlurFilterRasterNodeDefinition.handle);
    }
    
    addMotionBlurFilterRasterNode(motionBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addMotionBlurFilterRasterNode(this.handle, motionBlurFilterRasterNodeDefinition.handle);
    }
    
    addRadialBlurFilterRasterNode(radialBlurFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addRadialBlurFilterRasterNode(this.handle, radialBlurFilterRasterNodeDefinition.handle);
    }
    
    addLevelsAdjustmentRasterNode(levelsAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addLevelsAdjustmentRasterNode(this.handle, levelsAdjustmentRasterNodeDefinition.handle);
    }
    
    addBrightnessContrastAdjustmentRasterNode(brightnessContrastAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addBrightnessContrastAdjustmentRasterNode(this.handle, brightnessContrastAdjustmentRasterNodeDefinition.handle);
    }
    
    addShadowsHighlightsAdjustmentRasterNode(shadowsHighlightsAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addShadowsHighlightsAdjustmentRasterNode(this.handle, shadowsHighlightsAdjustmentRasterNodeDefinition.handle);
    }
    
    addBlackAndWhiteAdjustmentRasterNode(blackAndWhiteAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addBlackAndWhiteAdjustmentRasterNode(this.handle, blackAndWhiteAdjustmentRasterNodeDefinition.handle);
    }
    
    addRecolourAdjustmentRasterNode(recolourAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addRecolourAdjustmentRasterNode(this.handle, recolourAdjustmentRasterNodeDefinition.handle);
    }
    
    addPosteriseAdjustmentRasterNode(posteriseAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addPosteriseAdjustmentRasterNode(this.handle, posteriseAdjustmentRasterNodeDefinition.handle);
    }
    
    addSplitToningAdjustmentRasterNode(splitToningAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addSplitToningAdjustmentRasterNode(this.handle, splitToningAdjustmentRasterNodeDefinition.handle);
    }
    
    addInvertAdjustmentRasterNode(invertAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addInvertAdjustmentRasterNode(this.handle, invertAdjustmentRasterNodeDefinition.handle);
    }
    
    addThresholdAdjustmentRasterNode(thresholdAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addThresholdAdjustmentRasterNode(this.handle, thresholdAdjustmentRasterNodeDefinition.handle);
    }
    
    addClarityFilterRasterNode(clarityFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addClarityFilterRasterNode(this.handle, clarityFilterRasterNodeDefinition.handle);
    }
    
    addUnsharpMaskFilterRasterNode(unsharpMaskFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addUnsharpMaskFilterRasterNode(this.handle, unsharpMaskFilterRasterNodeDefinition.handle);
    }
    
    addHighPassFilterRasterNode(highPassFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addHighPassFilterRasterNode(this.handle, highPassFilterRasterNodeDefinition.handle);
    }
    
    addDenoiseFilterRasterNode(denoiseFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addDenoiseFilterRasterNode(this.handle, denoiseFilterRasterNodeDefinition.handle);
    }
    
    addDiffuseFilterRasterNode(diffuseFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addDiffuseFilterRasterNode(this.handle, diffuseFilterRasterNodeDefinition.handle);
    }
    
    addDustAndScratchFilterRasterNode(dustAndScratchFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addDustAndScratchFilterRasterNode(this.handle, dustAndScratchFilterRasterNodeDefinition.handle);
    }
    
    addAddNoiseFilterRasterNode(addNoiseFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addAddNoiseFilterRasterNode(this.handle, addNoiseFilterRasterNodeDefinition.handle);
    }
    
    addRippleFilterRasterNode(rippleFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addRippleFilterRasterNode(this.handle, rippleFilterRasterNodeDefinition.handle);
    }
    
    addTwirlFilterRasterNode(twirlFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addTwirlFilterRasterNode(this.handle, twirlFilterRasterNodeDefinition.handle);
    }
    
    addSphericalFilterRasterNode(sphericalFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addSphericalFilterRasterNode(this.handle, sphericalFilterRasterNodeDefinition.handle);
    }
    
    addPinchPunchFilterRasterNode(pinchPunchFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addPinchPunchFilterRasterNode(this.handle, pinchPunchFilterRasterNodeDefinition.handle);
    }
    
    addWhiteBalanceAdjustmentRasterNode(whiteBalanceAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addWhiteBalanceAdjustmentRasterNode(this.handle, whiteBalanceAdjustmentRasterNodeDefinition.handle);
    }
    
    addColourBalanceAdjustmentRasterNode(colourBalanceAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addColourBalanceAdjustmentRasterNode(this.handle, colourBalanceAdjustmentRasterNodeDefinition.handle);
    }
    
    addVibranceAdjustmentRasterNode(vibranceAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addVibranceAdjustmentRasterNode(this.handle, vibranceAdjustmentRasterNodeDefinition.handle);
    }
    
    addNormalsAdjustmentRasterNode(normalsAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addNormalsAdjustmentRasterNode(this.handle, normalsAdjustmentRasterNodeDefinition.handle);
    }
    
    addVignetteFilterRasterNode(vignetteFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addVignetteFilterRasterNode(this.handle, vignetteFilterRasterNodeDefinition.handle);
    }
    
    addDefringeFilterRasterNode(defringeFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addDefringeFilterRasterNode(this.handle, defringeFilterRasterNodeDefinition.handle);
    }
    
    addVoronoiFilterRasterNode(voronoiFilterRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addVoronoiFilterRasterNode(this.handle, voronoiFilterRasterNodeDefinition.handle);
    }
    
    addSelectiveColourAdjustmentRasterNode(selectiveColourAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addSelectiveColourAdjustmentRasterNode(this.handle, selectiveColourAdjustmentRasterNodeDefinition.handle);
    }
    
    addHSLShiftAdjustmentRasterNode(hslShiftAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addHSLShiftAdjustmentRasterNode(this.handle, hslShiftAdjustmentRasterNodeDefinition.handle);
    }
    
    addCurvesAdjustmentRasterNode(curvesAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addCurvesAdjustmentRasterNode(this.handle, curvesAdjustmentRasterNodeDefinition.handle);
    }

    addToneCompressionAdjustmentRasterNode(toneCompressionAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addToneCompressionAdjustmentRasterNode(this.handle, toneCompressionAdjustmentRasterNodeDefinition.handle);
    }
    
    addToneStretchAdjustmentRasterNode(toneStretchAdjustmentRasterNodeDefinition) {
        AddChildNodesCommandBuilderApi.addToneStretchAdjustmentRasterNode(this.handle, toneStretchAdjustmentRasterNodeDefinition.handle);
    }
    
    addContainerNode(containerNodeDefinition) {
        AddChildNodesCommandBuilderApi.addContainerNode(this.handle, containerNodeDefinition.handle);
    }
    
    createCommand(andSelect = true, childListType = NodeChildType.Main) {
        return new DocumentCommand(AddChildNodesCommandBuilderApi.createCommand(this.handle, andSelect, childListType));
    }
    
    set autoName(isAutoName) {
        AddChildNodesCommandBuilderApi.setAutoName(this.handle, isAutoName);
    }
    
    set forcePixelAlignment(forceAlignment) {
        AddChildNodesCommandBuilderApi.setForcePixelAlignment(this.handle, forceAlignment);
    }
    
    set clearCurrentSelection(clear) {
        AddChildNodesCommandBuilderApi.setClearCurrentSelection(this.handle, clear);
    }
    
    set appendCurrentSelection(append) {
        AddChildNodesCommandBuilderApi.setAppendCurrentSelection(this.handle, append);
    }
    
    setInsertionTarget(targetNode) {
        AddChildNodesCommandBuilderApi.setInsertionTarget(this.handle, targetNode.handle);
    }
    
    setInsertionMode(insertionMode) {
        AddChildNodesCommandBuilderApi.setInsertionMode(this.handle, insertionMode);
    }
    
    static create() {
        return new AddChildNodesCommandBuilder(AddChildNodesCommandBuilderApi.create());
    }
}

class SetHatchFillAttributesCommandBuilder extends HandleObject {
    static create() {
        return new SetHatchFillAttributesCommandBuilder(SetHatchFillAttributesCommandBuilderApi.create());
    }

    get [Symbol.toStringTag]() {
        return 'SetHatchFillAttributesCommandBuilder';
    }

    static create() {
        return new SetHatchFillAttributesCommandBuilder(SetHatchFillAttributesCommandBuilderApi.create());
    }

    reset() {
        return SetHatchFillAttributesCommandBuilderApi.reset(this.handle);
    }

    set pattern(value) {
        SetHatchFillAttributesCommandBuilderApi.setPattern(this.handle, value.handle);
    }

    set penColour(value) {
        SetHatchFillAttributesCommandBuilderApi.setPenColour(this.handle, value?.handle);
    }

    set brushColour(value) {
        SetHatchFillAttributesCommandBuilderApi.setBrushColour(this.handle, value?.handle);
    }

    set scale(value) {
        SetHatchFillAttributesCommandBuilderApi.setScale(this.handle, value);
    }

    set rotation(value) {
        SetHatchFillAttributesCommandBuilderApi.setRotation(this.handle, value);
    }

    set lineWeight(value) {
        SetHatchFillAttributesCommandBuilderApi.setLineWeight(this.handle, value);
    }

    set fixScale(value) {
        SetHatchFillAttributesCommandBuilderApi.setFixScale(this.handle, value);
    }

    set useRelativeTransform(value) {
        SetHatchFillAttributesCommandBuilderApi.setUseRelativeTransform(this.handle, value);
    }

    set useCurrentItemOriginForRelativeTransform(value) {
        SetHatchFillAttributesCommandBuilderApi.setUseCurrentItemOriginForRelativeTransform(this.handle, value);
    }

    createBrushFillCommand(selection, contentType = ContentType.Main, useTextSelection = false, applyToAllFills = false) {
        return new DocumentCommand(SetHatchFillAttributesCommandBuilderApi.createBrushFillCommand(this.handle, contentType, useTextSelection, selection?.handle, applyToAllFills));
    }

    createPenFillCommand(selection, contentType = ContentType.Main, useTextSelection = false, applyToAllFills = false) {
        return new DocumentCommand(SetHatchFillAttributesCommandBuilderApi.createPenFillCommand(this.handle, contentType, useTextSelection, selection?.handle, applyToAllFills));
    }

    createTransparencyFillCommand(selection, contentType = ContentType.Main, useTextSelection = false, applyToAllFills = false) {
        return new DocumentCommand(SetHatchFillAttributesCommandBuilderApi.createTransparencyFillCommand(this.handle, contentType, useTextSelection, selection?.handle, applyToAllFills));
    }
}

module.exports.Command = Command;
module.exports.DocumentCommand = DocumentCommand;
module.exports.CompoundCommandBuilder = CompoundCommandBuilder;
module.exports.AddChildNodesCommandBuilder = AddChildNodesCommandBuilder;
module.exports.SetHatchFillAttributesCommandBuilder = SetHatchFillAttributesCommandBuilder;
// convenience
module.exports.InsertionMode = InsertionMode;
module.exports.NodeMoveType = NodeMoveType;
module.exports.NodeChildType = NodeChildType;
module.exports.RasterSelectionLogicalOperation = RasterSelectionLogicalOperation;

