'use strict';

const { SpanCollection } = require('./collection.js');

// The highest codepoint that can be encoded with 1 byte in UTF-8
const UTF8_1_MAX = 0x7F;
// The highest codepoint that can be encoded with 2 bytes in UTF-8
const UTF8_2_MAX = 0x7FF;
// The highest codepoint that can be encoded with 3 bytes in UTF-8
const UTF8_3_MAX = 0xFFFF;
// The highest codepoint that can be encoded with 4 bytes in UTF-8
const UTF8_4_MAX = 0x10FFFF;
// If a character, masked with UTF8_CONTINUATION_MASK, matches this value, it is a UTF-8 continuation byte
const UTF8_CONTINUATION_VALUE = 0x80;
// The mask to a apply to a character before testing it against UTF8_CONTINUATION_VALUE
const UTF8_CONTINUATION_MASK = 0xC0;
// The number of bits of a codepoint that are contained in a UTF-8 continuation byte
const UTF8_CONTINUATION_CODEPOINT_BITS = 6;

const utf8_leading_bytes =
[
    // mask, value
    [ 0x80, 0x00 ], // 0xxxxxxx
    [ 0xE0, 0xC0 ], // 110xxxxx
    [ 0xF0, 0xE0 ], // 1110xxxx
    [ 0xF8, 0xF0 ]  // 11110xxx
];

function calculate_utf8_len(codepoint) {
    if (codepoint <= UTF8_1_MAX)
        return 1;

    if (codepoint <= UTF8_2_MAX)
        return 2;

    if (codepoint <= UTF8_3_MAX)
        return 3;

    return 4;
}

function encode_utf8(codepoint, buffer, len, index)
{
    const size = calculate_utf8_len(codepoint);
    
    // Not enough space left on the string
    if (index + size > len)
        return 0;

    // Write the continuation bytes in reverse order first
    for (let cont_index = size - 1; cont_index > 0; cont_index--)
    {
        let cont = codepoint & ~UTF8_CONTINUATION_MASK;
        cont |= UTF8_CONTINUATION_VALUE;

        buffer[index + cont_index] = cont;
        codepoint >>= UTF8_CONTINUATION_CODEPOINT_BITS;
    }

    // Write the leading byte
    let pattern = utf8_leading_bytes[size - 1];
    let lead = codepoint & ~(pattern[0]);
    lead |= pattern[1];
    buffer[index] = lead;

    return size;
}



class Buffer extends require('affinity:buffer').Buffer
{
    constructor(sz) {
        super(sz);
    }

    *[Symbol.iterator]() {
        for (const item of this.items) {
            yield item;
        }
    }

    get items() {
        return new SpanCollection(this);
    }

    concat(other) {
        const res = new Buffer(this.length + other.length);
        let i = 0;
        for (const item of this) {
            res[i++] = item;
        }
        for (const item of other) {
            res[i++] = item;
        }
        return res;
    }

    reverse() {
        for (let i = 0, j = this.length - 1; i < j; ++i, --j) {
            let tmp = this[i];
            this[i] = this[j];
            this[j] = tmp;
        }
        return this;
    }

    static utf16(str) {
        const buf = new Buffer(str.length * 2);
        const bufView = new Uint16Array(buf.buffer);
        for (let i = 0, strLen = str.length; i < strLen; i++) {
            bufView[i] = str.charCodeAt(i);
        }
        return buf;
    }

    static utf8(str) {
        const strLen = str.length;
        let utf8Len = 0;
        for (let i = 0; i < strLen; ++i) {
            utf8Len += calculate_utf8_len(str.codePointAt(i));
        }
        const buf = new Buffer(utf8Len);
        for (let i = 0, j = 0; i < strLen; ++i) {
            j += encode_utf8(str.codePointAt(i), buf, utf8Len, j);
        }
        return buf;
    }
}

module.exports.Buffer = Buffer;
